/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.lifecyle.healthcheck;

import static com.mulesoft.mule.runtime.gw.model.gatekeeper.status.GatekeeperStatus.AWAITING_CONTRACTS;
import static com.mulesoft.mule.runtime.gw.model.gatekeeper.status.GatekeeperStatus.AWAITING_POLICIES;
import static com.mulesoft.mule.runtime.gw.model.gatekeeper.status.GatekeeperStatus.AWAITING_TRACKING;
import static com.mulesoft.mule.runtime.gw.model.gatekeeper.status.GatekeeperStatus.AWAITING_UNBLOCK;
import static com.mulesoft.mule.runtime.gw.model.gatekeeper.status.GatekeeperStatus.POLICIES_FAILED;
import static com.mulesoft.mule.runtime.gw.model.gatekeeper.status.GatekeeperStatus.READY;
import static com.mulesoft.mule.runtime.gw.model.gatekeeper.status.GatekeeperStatus.UNTRACKED;
import static java.lang.String.format;
import static java.util.stream.Collectors.joining;

import org.mule.runtime.api.healthcheck.ReadyStatus;

import com.mulesoft.mule.runtime.gw.model.gatekeeper.status.GatekeeperStatus;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Stream;

public class TextReadyStatusFactory implements ReadyStatusFactory {

  private final Map<GatekeeperStatus, Function<Long, ReadyStatus>> factories;

  public TextReadyStatusFactory() {
    factories = new HashMap<>();
    factories.put(AWAITING_POLICIES, this::awaitingPolicies);
    factories.put(POLICIES_FAILED, this::policiesFailed);
    factories.put(AWAITING_CONTRACTS, this::awaitingContracts);
    factories.put(AWAITING_UNBLOCK, this::awaitingUnblock);
    factories.put(AWAITING_TRACKING, this::awaitingTracking);
    factories.put(UNTRACKED, this::untracked);
    factories.put(READY, this::ready);
  }

  @Override
  public ReadyStatus readyStatus(Long apiId, GatekeeperStatus gatekeeperStatus) {
    return factories.get(gatekeeperStatus).apply(apiId);
  }

  @Override
  public ReadyStatus missingApi(Long apiId) {
    return blockedStatus(apiId, "API was not deployed.");
  }

  @Override
  public ReadyStatus readyStatus(List<ReadyStatus> statusList) {
    if (statusList.size() == 1) {
      return statusList.get(0);
    }

    boolean readyStatus = statusList.stream().allMatch(ReadyStatus::isReady);

    String message = statusList.stream()
        .map(ReadyStatus::statusDescription)
        .flatMap(description -> description.map(Stream::of).orElseGet(Stream::empty))
        .sorted()
        .collect(joining("\n"));

    return new DefaultReadyStatus(readyStatus, appMessage(readyStatus) + message);
  }

  private String appMessage(boolean readyStatus) {
    return readyStatus ? "" : "Gatekeeper is blocking the app due to:\n";
  }

  protected ReadyStatus awaitingPolicies(Long apiId) {
    return blockedStatus(apiId, "Waiting for policies to be applied.");
  }

  protected ReadyStatus policiesFailed(Long apiId) {
    return blockedStatus(apiId, "Error applying policies.");
  }

  protected ReadyStatus awaitingContracts(Long apiId) {
    return blockedStatus(apiId, "Waiting for API's client applications to be successfully retrieved.");
  }

  protected ReadyStatus awaitingUnblock(Long apiId) {
    return blockedStatus(apiId, "Unblocking requirements met, API should be unblocked shortly.");
  }

  protected ReadyStatus awaitingTracking(Long apiId) {
    return blockedStatus(apiId, "Awaiting for successful API tracking.");
  }

  protected ReadyStatus untracked(Long apiId) {
    return blockedStatus(apiId, "API not found in the API Platform.");
  }

  protected ReadyStatus ready(Long apiId) {
    return new DefaultReadyStatus(true, format("API %s: Ready.", apiId));
  }

  protected ReadyStatus blockedStatus(Long apiId, String message) {
    return new DefaultReadyStatus(false, format("API %s: Not Ready. %s", apiId, message));
  }
}
