/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.offline;

import static com.mulesoft.mule.runtime.gw.api.logging.ExceptionDescriptor.errorMessage;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.apache.commons.lang3.StringUtils.substringBeforeLast;

import com.mulesoft.mule.runtime.gw.policies.OfflinePolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.factory.PolicyFactory;
import com.mulesoft.mule.runtime.gw.policies.lifecyle.PolicySetDeploymentListener;
import com.mulesoft.mule.runtime.gw.policies.serialization.OfflinePolicyDeserializationException;
import com.mulesoft.mule.runtime.gw.policies.serialization.PolicyDefinitionSerializer;
import com.mulesoft.mule.runtime.gw.policies.service.PolicyDeploymentService;

import java.io.File;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CopyOnWriteArrayList;

import org.apache.commons.io.monitor.FileAlterationListenerAdaptor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OfflinePolicyListener extends FileAlterationListenerAdaptor {

  private static final Logger LOGGER = LoggerFactory.getLogger(OfflinePolicyListener.class);

  private final PolicyDefinitionSerializer serializer;

  private final PolicyDeploymentService policyDeploymentService;

  private final List<PolicySetDeploymentListener> deploymentListeners;

  public OfflinePolicyListener(PolicyDeploymentService policyDeploymentService) {
    this.deploymentListeners = new CopyOnWriteArrayList<>();
    this.policyDeploymentService = policyDeploymentService;
    this.serializer = new PolicyDefinitionSerializer();
  }

  @Override
  public void onFileCreate(File definitionFile) {
    policyDefinition(definitionFile).ifPresent(policyDefinition -> {
      LOGGER.debug("New offline policy detected: {}", policyDefinition.getName());
      policyDeploymentService.newPolicy(policyDefinition);
    });
  }

  @Override
  public void onFileChange(File definitionFile) {
    policyDefinition(definitionFile).ifPresent(policyDefinition -> {
      LOGGER.debug("Modification detected on offline policy {}", policyDefinition.getName());
      policyDeploymentService.removePolicy(policyDefinition);
      policyDeploymentService.newPolicy(policyDefinition);
    });
  }

  @Override
  public void onFileDelete(File definitionFile) {
    String policyName = removeFileNameExtension(definitionFile);
    LOGGER.debug("Offline policy removal detected: {}", policyName);
    policyDeploymentService.removePolicy(policyName);
  }

  /**
   * Adds a listener which will be notified of the events related to policy deployments
   *
   * @param listener the listener to add
   */
  void addPolicyDeploymentListener(PolicySetDeploymentListener listener) {
    this.deploymentListeners.add(listener);
  }

  private Optional<OfflinePolicyDefinition> policyDefinition(File definitionFile) {

    try {
      OfflinePolicyDefinitionValidator.validate(definitionFile);
      return of(serializer.deserializeOfflineFromFile(definitionFile));
    } catch (OfflinePolicyDeserializationException e) {
      LOGGER.error("Error reading offline policy definition: " + errorMessage(e));
    }

    return empty();
  }

  private String removeFileNameExtension(File file) {
    return substringBeforeLast(file.getName(), ".");
  }

}
