/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.pointcut;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Lists.newArrayList;
import static org.mule.runtime.http.policy.api.SourcePolicyAwareAttributes.noAttributes;

import org.mule.runtime.policy.api.AttributeAwarePointcut;
import org.mule.runtime.policy.api.PolicyAwareAttributes;
import org.mule.runtime.policy.api.PolicyPointcut;
import org.mule.runtime.policy.api.PolicyPointcutParameters;

import java.util.List;

/**
 * Composition of different types of pointcuts
 */
public class CompositePointcut implements AttributeAwarePointcut {

  /**
   * Operator to use among defined pointcuts when matching
   */
  private final CompositePointcutOperator operator;

  /**
   * List of pointcuts to use in the matching
   */
  private List<PolicyPointcut> policyPointcuts;

  private CompositePointcut(CompositePointcutOperator operator, List<PolicyPointcut> policyPointcuts) {
    checkNotNull(operator, "Composite pointcut operator can not be null");
    checkNotNull(policyPointcuts, "Composite pointcuts can not be null");
    this.policyPointcuts = policyPointcuts;
    this.operator = operator;
  }

  public static CompositePointcut or(List<PolicyPointcut> pointcuts) {
    return new CompositePointcut(CompositePointcutOperator.OR, pointcuts);
  }

  public static CompositePointcut and(List<PolicyPointcut> pointcuts) {
    return new CompositePointcut(CompositePointcutOperator.AND, pointcuts);
  }

  public static CompositePointcut or(PolicyPointcut left, PolicyPointcut right) {
    return new CompositePointcut(CompositePointcutOperator.OR, newArrayList(left, right));
  }

  public static CompositePointcut and(PolicyPointcut left, PolicyPointcut right) {
    return new CompositePointcut(CompositePointcutOperator.AND, newArrayList(left, right));
  }

  @Override
  public boolean matches(PolicyPointcutParameters policyPointcutParameters) {
    return operator.equals(CompositePointcutOperator.OR)
        ? policyPointcuts.stream().anyMatch(pointcut -> pointcut.matches(policyPointcutParameters))
        : policyPointcuts.stream().allMatch(pointcut -> pointcut.matches(policyPointcutParameters));
  }

  @Override
  public PolicyAwareAttributes sourcePolicyAwareAttributes() {
    return policyPointcuts.stream().filter(p -> p instanceof AttributeAwarePointcut)
        .map(p -> ((AttributeAwarePointcut) p).sourcePolicyAwareAttributes())
        .reduce(noAttributes(), PolicyAwareAttributes::merge);
  }

  enum CompositePointcutOperator {
    OR, AND
  }

}
