/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.pointcut;

import static java.util.regex.Pattern.CASE_INSENSITIVE;

import org.mule.runtime.http.policy.api.HttpPolicyPointcutParameters;
import org.mule.runtime.http.policy.api.SourcePolicyAwareAttributes;
import org.mule.runtime.policy.api.AttributeAwarePointcut;
import org.mule.runtime.policy.api.PolicyAwareAttributes;
import org.mule.runtime.policy.api.PolicyPointcutParameters;

import java.io.Serializable;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EffectiveHttpResourcePointcut implements AttributeAwarePointcut, Serializable {

  private static final Logger LOGGER = LoggerFactory.getLogger(EffectiveHttpResourcePointcut.class);
  private static final long serialVersionUID = -5003310196710218800L;

  /**
   * Regular expression to match against the HTTP request's path
   */
  private final Pattern pathPattern;

  /**
   * Regular expression to match against the HTTP request's method
   */
  private final Pattern methodPattern;

  /**
   * Extractor of the path of the HTTP request that must be used to do the pointcut matching
   */
  private final PolicyPointcutParametersPathExtractor pathExtractor;

  /**
   * Name of the policy to which the pointcut belongs
   */
  private final String policyName;

  public EffectiveHttpResourcePointcut(String policyName, String pathRegex, String methodRegex,
                                       PolicyPointcutParametersPathExtractor pathExtractor) {
    this.policyName = policyName;
    this.pathPattern = Pattern.compile(pathRegex, CASE_INSENSITIVE);
    this.methodPattern = Pattern.compile(methodRegex, CASE_INSENSITIVE);
    this.pathExtractor = pathExtractor;
  }

  @Override
  public boolean matches(PolicyPointcutParameters policyPointcutParameters) {
    PolicyPointcutParameters parameters = policyPointcutParameters.getSourceParameters().orElse(policyPointcutParameters);

    try {
      String path = pathExtractor.getPath(parameters);
      String method = ((HttpPolicyPointcutParameters) parameters).getMethod();

      boolean matches = methodPattern.matcher(method).matches() && pathPattern.matcher(path).matches();

      if (!matches) {
        LOGGER.trace("Request {{}, {}} does not match the following resource pointcut: {} for policy {}", path, method, this,
                     policyName);
      } else {
        LOGGER.trace("Request {{}, {}} matches the following resource pointcut: {} for policy {}", path, method, this,
                     policyName);
      }

      return matches;
    } catch (ClassCastException e) {
      LOGGER.trace("Invalid parameters type found. {}", e.getMessage());
    }

    return false;
  }

  @Override
  public PolicyAwareAttributes sourcePolicyAwareAttributes() {
    return new SourcePolicyAwareAttributes.Builder().requestPathPatterns(pathPattern).build();
  }

  public String getPath() {
    return pathPattern.toString();
  }

  public String getMethod() {
    return methodPattern.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    EffectiveHttpResourcePointcut that = (EffectiveHttpResourcePointcut) o;

    return pathPattern.toString().equals(that.pathPattern.toString()) && methodPattern.toString()
        .equals(that.methodPattern.toString());
  }

  @Override
  public int hashCode() {
    int result = pathPattern.toString().hashCode();
    result = 31 * result + (methodPattern.toString().hashCode());
    return result;
  }

  @Override
  public String toString() {
    return "{path=" + pathPattern + ", method=" + methodPattern + "}";
  }
}
