/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.pointcut;

import static com.google.common.base.Preconditions.checkNotNull;

import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.policy.api.HttpPolicyPointcutParameters;
import org.mule.runtime.http.policy.api.SourcePolicyAwareAttributes;
import org.mule.runtime.policy.api.AttributeAwarePointcut;
import org.mule.runtime.policy.api.PolicyAwareAttributes;
import org.mule.runtime.policy.api.PolicyPointcutParameters;

import java.io.Serializable;
import java.util.List;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class HttpHeaderPointcut implements AttributeAwarePointcut, Serializable {

  private static final Logger LOGGER = LoggerFactory.getLogger(HttpHeaderPointcut.class);

  private static final long serialVersionUID = 4882453834999985655L;

  /**
   * Header name to match against the HTTP request headers
   */
  private final String name;

  /**
   * Regular expression to match against the HTTP header values of the matching header name
   */
  private final Pattern valuePattern;

  /**
   * Name of the policy to which the pointcut belongs
   */
  private final String policyName;

  public HttpHeaderPointcut(String policyName, String name, String valuePattern) {
    checkNotNull(name);
    checkNotNull(valuePattern);
    this.name = name;
    this.valuePattern = Pattern.compile(valuePattern);
    this.policyName = policyName;
  }

  @Override
  public boolean matches(PolicyPointcutParameters parameters) {
    parameters = parameters.getSourceParameters().orElse(parameters);
    try {
      MultiMap<String, String> headers = ((HttpPolicyPointcutParameters) parameters).getHeaders();
      List<String> allValues = headers.getAll(name);
      boolean matches = allValues.stream().anyMatch(v -> valuePattern.matcher(v).matches());
      if (LOGGER.isTraceEnabled()) {
        if (!matches) {
          LOGGER.trace("Request {{}, {}} does not match the following header pointcut: {} for policy {}", name,
                       allValues.isEmpty() ? "<Not Present>" : allValues, this, policyName);
        } else {
          LOGGER.trace("Request {{}, {}} matches the following header pointcut: {} for policy {}", name, allValues, this,
                       policyName);
        }
      }
      return matches;
    } catch (ClassCastException e) {
      LOGGER.trace("Invalid parameters type found. {}", e.getMessage());
    }
    return false;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    HttpHeaderPointcut that = (HttpHeaderPointcut) o;

    return name.equals(that.name) && valuePattern.toString()
        .equals(that.valuePattern.toString());
  }

  @Override
  public int hashCode() {
    int result = name.hashCode();
    result = 31 * result + (valuePattern.toString().hashCode());
    return result;
  }

  @Override
  public String toString() {
    return "{name=" + name + ", value=" + valuePattern + "}";
  }

  @Override
  public PolicyAwareAttributes sourcePolicyAwareAttributes() {
    return new SourcePolicyAwareAttributes.Builder().headers(name).build();
  }
}
