/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.pointcut;

import static com.mulesoft.mule.runtime.gw.api.annotations.GatewayAppAnnotations.DISABLE_POLICIES_ANNOTATION;
import static com.mulesoft.mule.runtime.gw.api.policy.HttpResourcePointcut.HTTP_LISTENER_COMPONENT_IDENTIFIER;
import static com.mulesoft.mule.runtime.gw.api.policy.HttpResourcePointcut.HTTP_REQUEST_COMPONENT_IDENTIFIER;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.core.internal.util.OneTimeWarning;
import org.mule.runtime.policy.api.PolicyPointcut;
import org.mule.runtime.policy.api.PolicyPointcutParameters;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class IsHttpComponentPointcut implements PolicyPointcut {

  private static final Logger LOGGER = LoggerFactory.getLogger(IsHttpComponentPointcut.class);

  private final OneTimeWarning notApplicableWarn = new OneTimeWarning(LOGGER,
                                                                      "{} is annotated with 'api-gateway:disablePolicies', no operation block policies will be applied to this element.");

  /**
   * Name of the policy to which the pointcut belongs
   */
  private final String policyName;

  public IsHttpComponentPointcut(String policyName) {
    this.policyName = policyName;
  }

  @Override
  public boolean matches(PolicyPointcutParameters policyPointcutParameters) {
    ComponentIdentifier componentIdentifier =
        policyPointcutParameters.getComponent().getLocation().getComponentIdentifier().getIdentifier();

    boolean isRequester = HTTP_REQUEST_COMPONENT_IDENTIFIER.equals(componentIdentifier);
    boolean matches = isRequester || HTTP_LISTENER_COMPONENT_IDENTIFIER.equals(componentIdentifier);

    if (isAnnotated(policyPointcutParameters)) {
      if (isRequester) {
        notApplicableWarn.warn(componentIdentifier);
        LOGGER.trace("{} does not apply for pointcut for being annotated with 'api-gateway:disablePolicies'. " +
            "No policies will be applied to this operation.", componentIdentifier);
        matches = false;
      } else {
        LOGGER.trace("Annotation 'api-gateway:disablePolicies' has no effect on component {}", componentIdentifier);
      }
    }

    if (matches) {
      LOGGER.trace("Request matches pointcut of policy {} since {} is a valid HTTP component.", policyName, componentIdentifier);
    } else {
      LOGGER.trace("Request does not match pointcut of policy {} since {} is not a valid HTTP component.", policyName,
                   componentIdentifier);
    }
    return matches;
  }

  private boolean isAnnotated(PolicyPointcutParameters policyPointcutParameters) {
    String annotation = (String) policyPointcutParameters.getComponent().getAnnotation(DISABLE_POLICIES_ANNOTATION);
    return annotation != null && annotation.equals("true");
  }
}
