/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.serialization;

import static com.fasterxml.jackson.annotation.JsonInclude.Include.NON_NULL;
import static com.fasterxml.jackson.databind.DeserializationFeature.ACCEPT_SINGLE_VALUE_AS_ARRAY;
import static com.fasterxml.jackson.databind.SerializationFeature.INDENT_OUTPUT;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.substringBeforeLast;

import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.api.policy.HttpResourcePointcut;
import com.mulesoft.mule.runtime.gw.api.policy.PolicyTemplateKey;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.OfflinePolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.serialization.PolicyDefinitionDto.APIKeyDTO;
import com.mulesoft.mule.runtime.gw.policies.serialization.PolicyDefinitionDto.PolicyTemplateKeyDTO;

import com.fasterxml.jackson.databind.ObjectMapper;

import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

public class PolicyDefinitionSerializer {

  private ObjectMapper objectMapper;

  public PolicyDefinitionSerializer() {
    this.objectMapper = new ObjectMapper();
    objectMapper.setSerializationInclusion(NON_NULL);
    objectMapper.enable(ACCEPT_SINGLE_VALUE_AS_ARRAY);
    objectMapper.enable(INDENT_OUTPUT);
  }

  /**
   * Serializes a policy definition to JSON format and stores the output in a file
   *
   * @param policyDefinition the definition to serialize
   * @param destFile the file where to store the JSON
   */
  public void serializeToFile(PolicyDefinition policyDefinition, File destFile) throws IOException {
    FileWriter fileWriter = new FileWriter(destFile);
    objectMapper.writeValue(fileWriter, PolicyDefinitionDto.from(policyDefinition));
  }

  /**
   * Deserializes the JSON stored in a file into a {@link PolicyDefinition}.
   *
   * @param file the file where to read the JSON
   * @return the policy definition
   */
  public PolicyDefinition deserializeFromFile(File file) throws IOException {
    PolicyDefinitionDto dto = objectMapper.readValue(new FileReader(file), PolicyDefinitionDto.class);

    List<HttpResourcePointcut> resourcePointcuts = resourcePointcuts(dto);

    PolicyConfiguration policyConfiguration = new PolicyConfiguration(dto.getConfiguration(), dto.getConfigurationVersion());

    return dto.getOnline()
        ? new PolicyDefinition(dto.getId(), adaptDtoKey(dto.getTemplateKey()), apiKeys(dto.getApis()), resourcePointcuts,
                               dto.getOrder(), policyConfiguration)
        : new OfflinePolicyDefinition(dto.getId(), adaptDtoKey(dto.getTemplateKey()), apiKeys(dto.getApis()), resourcePointcuts,
                                      dto.getOrder(), policyConfiguration);
  }

  /**
   * Deserializes the JSON stored in a file into an {@link OfflinePolicyDefinition}.
   *
   * @param file the file where to read the JSON
   * @return the offline policy definitions
   */
  public OfflinePolicyDefinition deserializeOfflineFromFile(File file) {
    try {
      PolicyDefinitionDto dto = objectMapper.readValue(new FileReader(file), PolicyDefinitionDto.class);


      validateOfflineDefinition(dto);

      List<HttpResourcePointcut> resourcePointcuts = resourcePointcuts(dto);

      PolicyConfiguration policyConfiguration = new PolicyConfiguration(dto.getConfiguration());

      return new OfflinePolicyDefinition(removeFileNameExtension(file), adaptDtoKey(dto.getTemplateKey()), apiKeys(dto.getApis()),
                                         resourcePointcuts, dto.getOrder(), policyConfiguration);
    } catch (IOException e) {
      throw new OfflinePolicyDeserializationException("Cannot read offline policy definition from " + file, e);
    }
  }

  private List<HttpResourcePointcut> resourcePointcuts(PolicyDefinitionDto dto) {
    return dto.getPointcuts() != null ? dto.getPointcuts().getResources().stream()
        .map(resourceDTO -> new HttpResourcePointcut(resourceDTO.getPath(), resourceDTO.getMethod()))
        .collect(toList()) : new ArrayList<>();
  }

  private List<ApiKey> apiKeys(List<APIKeyDTO> apiKeyDTOs) {
    return apiKeyDTOs.stream().map(apiKeyDTO -> new ApiKey(apiKeyDTO.getId())).collect(toList());
  }

  private PolicyTemplateKey adaptDtoKey(PolicyTemplateKeyDTO templateKey) {
    return new PolicyTemplateKey(templateKey.getGroupId(), templateKey.getAssetId(), templateKey.getVersion());
  }

  private String removeFileNameExtension(File file) {
    return substringBeforeLast(file.getName(), ".");
  }

  private void validateOfflineDefinition(PolicyDefinitionDto dto) {
    validateTemplateKey(dto.getTemplateKey());

    if (dto.getApis() == null) {
      throw new OfflinePolicyDeserializationException("At least one API must be specified");
    }

    dto.getApis().forEach(this::validateApi);

    if (dto.getOrder() <= 0) {
      throw new OfflinePolicyDeserializationException("Order has to be a positive integer");
    }
  }

  private void validateTemplateKey(PolicyTemplateKeyDTO templateKey) {
    if (templateKey == null) {
      throw new OfflinePolicyDeserializationException("Template can not be empty");
    }

    if (isBlank(templateKey.getAssetId()) || isBlank(templateKey.getGroupId()) || isBlank(templateKey.getVersion())) {
      throw new OfflinePolicyDeserializationException("Template's assetId, groupId and version must be present");
    }
  }

  private void validateApi(APIKeyDTO api) {
    if (api == null) {
      throw new OfflinePolicyDeserializationException("API can not be empty");
    }

    if (api.getId() == null) {
      throw new OfflinePolicyDeserializationException("API's id must be present");
    }
  }

}
