/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.service;

import static java.util.stream.Collectors.toList;

import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.deployment.ApiService;
import com.mulesoft.mule.runtime.gw.logging.GatewayMuleAppLoggerFactory;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.ApiImplementation;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.model.PolicySet;
import com.mulesoft.mule.runtime.gw.policies.PolicyDefinitionBuilder;
import com.mulesoft.mule.runtime.gw.policies.PolicyDefinitionDeploymentStatus;
import com.mulesoft.mule.runtime.gw.policies.PolicyDeploymentStatus;
import com.mulesoft.mule.runtime.gw.policies.service.detection.PolicyChangeProcessor;
import com.mulesoft.mule.runtime.gw.policies.lifecyle.PolicySetDeploymentListener;
import com.mulesoft.mule.runtime.gw.policies.store.PolicyStore;
import com.mulesoft.anypoint.retry.RunnableRetrier;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;

import org.slf4j.Logger;

public class DefaultPolicySetDeploymentService implements PolicySetDeploymentService {

  private static final Logger LOGGER = GatewayMuleAppLoggerFactory.getLogger(DefaultPolicySetDeploymentService.class);

  private final RunnableRetrier<ApiKey> runnableRetrier;
  private final PolicyDeploymentService policyDeploymentService;
  private final PolicyDeploymentTracker policyDeploymentTracker;
  private final PolicyStore policyStore;
  private final List<PolicySetDeploymentListener> deploymentListeners;
  private final ApiService apiService;

  public DefaultPolicySetDeploymentService(RunnableRetrier<ApiKey> policySetRetrier,
                                           PolicyDeploymentService policyDeploymentService,
                                           PolicyDeploymentTracker policyDeploymentTracker,
                                           PolicyStore policyStore, ApiService apiService) {
    this.deploymentListeners = new CopyOnWriteArrayList<>();
    this.runnableRetrier = policySetRetrier;
    this.policyDeploymentService = policyDeploymentService;
    this.policyDeploymentTracker = policyDeploymentTracker;
    this.policyStore = policyStore;
    this.apiService = apiService;
  }

  /**
   * {@inheritDoc}
   * <p>
   * This implementation is synchronized to avoid a race condition in the case that this method is called a second time for the
   * same API when the processing of the first one has not finished. This is a real possibility since the callers are not
   * synchronized
   */
  @Override
  public synchronized void policiesForApi(ApiKey apiKey, PolicySet policySet) {
    PolicySet apiPolicySet = apiPolicySet(apiKey, policySet);
    runnableRetrier.scheduleRetry(apiKey, () -> {
      logPoliciesForApi(apiKey, apiPolicySet);

      new PolicyChangeProcessor(policyDeploymentService)
          .process(policyDeploymentTracker.onlinePolicyStatuses(apiKey), apiPolicySet.getPolicyDefinitions());

      if (failedDownloadTemplates(apiKey).size() == 0) {
        notifyPolicySetDeployed(apiKey, apiPolicySet);
      } else {
        LOGGER.debug("Template download failed for API {} - Policies {}.", apiKey, names(failedDownloadTemplates(apiKey)));
        throw new RuntimeException("There are still failures in template downloading");
      }
    });
  }

  @Override
  public void removeAll(ApiKey apiKey) {
    LOGGER.debug("Removing all policies from API {}", apiKey);

    // Notify listeners first, so they act before the API is not protected anymore
    notifyAllPoliciesRemoved(apiKey);

    List<PolicyDefinition> policiesToRemove = policyDeploymentTracker.apiRemoved(apiKey);
    policiesToRemove.forEach(policyDeploymentService::removePolicy);
  }

  @Override
  public void onApiDeploymentSuccess(Api api) {
    // Deploy offline policies to API
    policyStore
        .offlinePolicies().stream()
        .filter(policyDefinition -> policyDefinition.getApiKeys().contains(api.getKey()))
        .forEach(policyDefinition -> policyDeploymentService.newPolicy(specializeToApi(api.getKey(), policyDefinition)));
  }

  @Override
  public void onApiUndeploymentStart(ApiImplementation implementation) {
    removeAll(implementation.getApiKey());
  }

  @Override
  public void onApiRedeploymentStart(ApiImplementation implementation) {
    notifyAllPoliciesRemoved(implementation.getApiKey());
    policyDeploymentTracker.apiRemoved(implementation.getApiKey());
  }

  @Override
  public void addPolicyDeploymentListener(PolicySetDeploymentListener listener) {
    deploymentListeners.add(listener);
  }

  @Override
  public void conciliatePolicies(ApiKey apiKey, List<PolicyDefinition> desiredPolicies) {
    List<PolicyDefinition> actualPolicies = filterByApi(policyStore.onlinePolicies(), apiKey);

    actualPolicies.stream()
        .filter(policyDefinition -> !desiredPolicies.contains(policyDefinition))
        .forEach(policyDefinition -> policyStore.remove(policyDefinition.getName()));

    desiredPolicies.stream()
        .filter(policyDefinition -> !actualPolicies.contains(policyDefinition))
        .forEach(policyDefinition -> policyStore.store(policyDefinition)); // Template will be resolved when deployed
  }

  @Override
  public Map<ApiKey, List<PolicyDefinition>> storedOnlinePoliciesByApi() {
    Map<ApiKey, List<PolicyDefinition>> groupedPolicies = new HashMap<>();

    policyStore.onlinePolicies().forEach(policyDefinition -> policyDefinition.getApiKeys().forEach(apiKey -> {
      groupedPolicies.putIfAbsent(apiKey, new ArrayList<>());
      groupedPolicies.get(apiKey).add(policyDefinition);
    }));

    return groupedPolicies;
  }

  /**
   * Specializes the policy set to an specific api
   *
   * @param apiKey the key of the api
   * @param policySet the policy set to trim
   * @return the policy set relevant only to the apikey
   */
  private PolicySet apiPolicySet(ApiKey apiKey, PolicySet policySet) {
    List<PolicyDefinition> apiDefinitions = policySet.getPolicyDefinitions().stream()
        .filter(definition -> definition.getApiKeys().contains(apiKey))
        .map(definition -> specializeToApi(apiKey, definition))
        .collect(toList());
    return new PolicySet(apiDefinitions, policySet.getOrigin());
  }

  /**
   * @return the policy definition where the apikey is the only api it applies to.
   */
  private PolicyDefinition specializeToApi(ApiKey apiKey, PolicyDefinition definition) {
    return new PolicyDefinitionBuilder(definition).apiKey(apiKey).build();
  }

  private void logPoliciesForApi(ApiKey apiKey, PolicySet policySet) {
    if (LOGGER.isDebugEnabled()) {
      List<String> policyNames = policySet.getPolicyDefinitions().stream().map(PolicyDefinition::getName).collect(toList());
      LOGGER
          .debug("Deploying policies {} from {} to API {}", policyNames, policySet.isFromPlatform() ? "Platform" : "File System",
                 apiKey);
    }
  }

  private void notifyAllPoliciesRemoved(ApiKey apiKey) {
    deploymentListeners.forEach(policyDeploymentListener -> {
      try {
        policyDeploymentListener.onPoliciesRemoved(apiKey);
      } catch (Exception e) {
        LOGGER.warn("Error on polices removed listener: {}", e.getMessage());
      }
    });
  }

  private void notifyPolicySetDeployed(ApiKey apiKey, PolicySet policySet) {
    deploymentListeners.forEach(policyDeploymentListener -> {
      try {
        policyDeploymentListener
            .onPolicySetDeploymentCompleted(apiKey, policySet, policyDeploymentTracker.onlinePolicyStatuses(apiKey)
                .stream().map(PolicyDeploymentStatus::getLatestPolicyStatus).collect(toList()));
      } catch (Exception e) {
        LOGGER.warn("Error on policy deployment completed listener: {}", e.getMessage());
      }
    });
  }

  private List<PolicyDefinition> filterByApi(List<PolicyDefinition> policyDefinitions, ApiKey apiKey) {
    return policyDefinitions.stream()
        .filter(policyDefinition -> policyDefinition.getApiKeys().contains(apiKey))
        .collect(toList());
  }

  private List<String> names(List<PolicyDefinitionDeploymentStatus> policyDeploymentStatuses) {
    return policyDeploymentStatuses.stream().map(status -> status.getPolicyDefinition().getName()).collect(toList());
  }

  private List<PolicyDefinitionDeploymentStatus> failedDownloadTemplates(ApiKey apiKey) {
    return policyDeploymentTracker
        .onlinePolicyStatuses(apiKey).stream()
        .map(PolicyDeploymentStatus::getLatestPolicyStatus)
        .filter(PolicyDefinitionDeploymentStatus::isTemplateDownloadFailed)
        .collect(toList());
  }
}
