/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.service.detection;

import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.PolicyDefinitionDeploymentStatus;
import com.mulesoft.mule.runtime.gw.policies.PolicyDeploymentStatus;
import com.mulesoft.mule.runtime.gw.policies.service.detection.change.ChangeType;
import com.mulesoft.mule.runtime.gw.policies.service.detection.change.PolicyAdded;
import com.mulesoft.mule.runtime.gw.policies.service.detection.change.PolicyRemoved;

import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

/**
 * A class that can discern, given the current policy status and the incoming information of what policies should be applied which
 * actions should be taken for each policy
 */
public class PolicySetChangeInspector {

  private PolicyChangeInspector changeInspector = new PolicyChangeInspector();

  /**
   * Given the current deployment statuses of all policies and the desired policy definitions determines which actions should be
   * taken to consolidate the current status with with the desired status.
   *
   * @param currentStatusList a {@link List} of {@link PolicyDefinitionDeploymentStatus} that reflects the current status.
   * @param newDefinitions a {@link List} of {@link PolicyDefinition} that should be currently applied.
   */
  public List<ChangeType> detectChanges(List<PolicyDeploymentStatus> currentStatusList, List<PolicyDefinition> newDefinitions) {
    List<ChangeType> changes = new LinkedList<>();

    for (PolicyDefinition definition : newDefinitions) {
      Optional<PolicyDeploymentStatus> currentStatus = currentStatusList.stream()
          .filter(status -> samePolicyId(status, definition)).findFirst();

      if (!currentStatus.isPresent()) {
        changes.add(new PolicyAdded(definition));
      } else {
        changes.add(changeInspector.detectChange(currentStatus.get(), definition));
      }
    }

    policiesToRemove(currentStatusList, newDefinitions).forEach(changes::add);

    return changes;
  }

  private Stream<ChangeType> policiesToRemove(List<PolicyDeploymentStatus> currentStatuses,
                                              List<PolicyDefinition> policySet) {
    return currentStatuses.stream()
        .filter(status -> policySet.stream().noneMatch(policyDefinition -> samePolicyId(status, policyDefinition)))
        .map(status -> status.getLatestPolicyStatus().getPolicyDefinition())
        .map(PolicyRemoved::new);
  }

  private boolean samePolicyId(PolicyDeploymentStatus status, PolicyDefinition policyDefinition) {
    return policyDefinition.getId().equals(status.getLatestPolicyStatus().getPolicyDefinition().getId());
  }
}
