/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.store;

import com.mulesoft.mule.runtime.gw.api.key.ApiKey;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.Policy;

import java.io.File;
import java.util.List;

public interface PolicyStore {

  /**
   * Returns every stored Policy definition
   *
   * @return Collection containing every stored policy definition
   */
  List<PolicyDefinition> load();

  /**
   * Stores just the policy definition.
   *
   * @param policyDefinition policy definition to store
   */
  void store(PolicyDefinition policyDefinition);

  /**
   * Stores the complete policy.
   *
   * @param policy policy to store
   */
  void store(Policy policy);

  /**
   * Removes the specified policy, valid or failed, from the store.
   *
   * @param policyName name of the policy to remove
   * @return true or false whether it was able to remove it or not
   */
  boolean remove(String policyName);

  /**
   * Reads and returns every online policy that is present in the store
   *
   * @return map of the present policies
   */
  List<PolicyDefinition> onlinePolicies();

  /**
   * Reads and returns every offline policy that is present in the store
   *
   * @return map of the present policies
   */
  List<PolicyDefinition> offlinePolicies();

  /**
   * Checks whether the specified policy is present in the store
   *
   * @param policyName name of the policy to check
   * @return true or false whether the specified policy is present or not
   */
  boolean contains(String policyName);

  /**
   * Returns the physical config file of the specified policy
   * 
   * @param policyName name of the policy
   * @return the config file
   */
  File getPolicyConfigFile(String policyName);

  /**
   * Returns the properties file of the specified policy
   *
   * @param policyName name of the policy
   * @return the properties file
   */
  File getEncryptedPropertiesFile(String policyName);

  /**
   * Stores the deployment failure exception in the failed policies folder
   * 
   * @param policyDefinition the policy that failed
   * @param apiKey the API where the policy deployment was being deployed
   * @param exception the failure
   */
  void storeDeploymentFailure(PolicyDefinition policyDefinition, ApiKey apiKey, Exception exception);

  /**
   * Deletes the stored deployment failure log of a policy to a specific API
   * 
   * @param policyDefinition the policy that failed
   * @param apiKey the API where the policy deployment was being deployed
   */
  void cleanDeploymentFailure(PolicyDefinition policyDefinition, ApiKey apiKey);

  /**
   * Deletes every stored deployment failure log of a policy
   * 
   * @param policyName name of the policy
   */
  void cleanDeploymentFailures(String policyName);
}
