/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.template.provider;

import static com.mulesoft.mule.runtime.gw.api.PolicyFolders.getPolicyTemplatesTempFolder;
import static java.util.Optional.empty;
import static org.apache.commons.io.FileUtils.deleteQuietly;
import static org.mule.runtime.core.api.util.FileUtils.unzip;

import com.mulesoft.mule.runtime.gw.logging.GatewayMuleAppLoggerFactory;
import com.mulesoft.mule.runtime.gw.model.PolicySpecification;
import com.mulesoft.mule.runtime.gw.model.ValidPolicySpecification;
import com.mulesoft.mule.runtime.gw.policies.serialization.PolicySpecificationBuilder;
import org.mule.runtime.deployment.model.api.policy.PolicyTemplateDescriptor;
import org.mule.runtime.module.deployment.impl.internal.policy.PolicyTemplateDescriptorFactory;

import com.mulesoft.mule.runtime.gw.api.policy.PolicyTemplateKey;
import com.mulesoft.mule.runtime.gw.policies.template.PolicyTemplate;
import com.mulesoft.mule.runtime.gw.policies.template.exception.PolicyTemplateAssetException;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;

public abstract class PolicyTemplateProvider {

  static final Logger LOGGER = GatewayMuleAppLoggerFactory.getLogger(PolicyTemplateProvider.class);

  private static final String TEMPLATE_FILE_NAME = "template.xml";

  protected final File templatesTempFolder;
  private final Map<PolicyTemplateKey, PolicyTemplate> policyTemplateMap = new HashMap<>();
  private final PolicyTemplateDescriptorFactory templateDescriptorFactory = new PolicyTemplateDescriptorFactory();

  public PolicyTemplateProvider() {
    templatesTempFolder = getPolicyTemplatesTempFolder();
  }

  /**
   * Provides a {@link PolicyTemplate} from the policy templates folder (M2_HOME/policies/policy-templates)
   * 
   * @param templateKey id of the template to load
   * @return loaded policy template
   */
  public synchronized PolicyTemplate provide(PolicyTemplateKey templateKey) {
    PolicyTemplate policyTemplate = policyTemplateMap.get(templateKey);
    PolicyTemplateAssets templateAssets = null;

    if (policyTemplate == null || !policyTemplate.getTemplateDescriptor().getRootFolder().exists()) {
      File templateExplodedFolder = new File(templatesTempFolder, templateKey.getName());

      try {
        templateAssets = getPolicyTemplateAssets(templateKey);
        unzip(templateAssets.getTemplateJarFile(), templateExplodedFolder);
      } catch (IOException e) {
        if (templateAssets != null && deleteQuietly(templateAssets.getTemplateJarFile())) {
          LOGGER.debug("Policy template JAR {} was deleted successfully", templateAssets.getTemplateJarFile().getName());
        }

        throw new PolicyTemplateAssetException("Error processing policy template file for " + templateKey.getName(), e);
      }

      PolicyTemplateDescriptor templateDescriptor = templateDescriptorFactory.create(templateExplodedFolder, empty());
      PolicySpecification policySpecification = new PolicySpecificationBuilder()
          .withFile(templateAssets.getTemplateYamlFile())
          .build();

      File templateFile = new File(templateExplodedFolder, TEMPLATE_FILE_NAME);
      policyTemplate = new PolicyTemplate(templateKey, templateFile, policySpecification, templateDescriptor);

      policyTemplateMap.put(templateKey, policyTemplate);
    }

    return policyTemplate;
  }

  protected abstract PolicyTemplateAssets getPolicyTemplateAssets(PolicyTemplateKey templateKey) throws IOException;
}
