/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.service.detection;

import static com.mulesoft.mule.runtime.gw.policies.PolicyDefinitionBuilder.hasIdentityManagementProperties;

import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.PolicyDefinitionBuilder;
import com.mulesoft.mule.runtime.gw.policies.PolicyDefinitionDeploymentStatus;
import com.mulesoft.mule.runtime.gw.policies.PolicyDeploymentStatus;
import com.mulesoft.mule.runtime.gw.policies.service.detection.change.ChangeType;
import com.mulesoft.mule.runtime.gw.policies.service.detection.change.PolicyAdded;
import com.mulesoft.mule.runtime.gw.policies.service.detection.change.PolicyUpdate;
import com.mulesoft.mule.runtime.gw.policies.service.detection.change.PolicyReorder;
import com.mulesoft.mule.runtime.gw.policies.service.detection.change.PolicyRevert;
import com.mulesoft.mule.runtime.gw.policies.service.detection.change.PolicyUnmodified;

import java.util.Optional;
import java.util.function.BiFunction;
import java.util.function.Function;

/**
 * A class that can discern, given the current policy status and the incoming information of what policies should be applied which
 * actions should be taken for each policy
 */
public class PolicyChangeInspector {

  /**
   * Given the current deployment status and a policy definition determines which action should be taken to consolidate the
   * current status with with the desired status.
   *
   * @param status a {@link PolicyDefinitionDeploymentStatus} that reflects the current status.
   * @param newDefinition a {@link PolicyDefinition} that should be currently applied.
   */
  public ChangeType detectChange(PolicyDeploymentStatus status, PolicyDefinition newDefinition) {
    if (isPolicyOrderUpdate(status, newDefinition)) {
      return new PolicyReorder(status.getAppliedPolicyStatus().get().getPolicyDefinition(), newDefinition);
    } else if (isPolicyRevert(status, newDefinition)) {
      return new PolicyRevert(newDefinition);
    } else if (isTemplateDownloadFailed(status)) {
      if (isApplied(status)) {
        return new PolicyUpdate(status.getAppliedPolicyStatus().get().getPolicyDefinition(), newDefinition);
      } else {
        return new PolicyAdded(newDefinition);
      }
    } else if (isPolicyUpdate(status, newDefinition)) {
      if (isApplied(status)) {
        return new PolicyUpdate(status.getAppliedPolicyStatus().get().getPolicyDefinition(), newDefinition);
      } else {
        return new PolicyAdded(newDefinition);
      }
    } else {
      return new PolicyUnmodified(newDefinition);
    }
  }

  private boolean isApplied(PolicyDeploymentStatus status) {
    return status.getAppliedPolicyStatus().isPresent();
  }

  private boolean isTemplateDownloadFailed(PolicyDeploymentStatus status) {
    return status.getLatestPolicyStatus().isTemplateDownloadFailed();
  }

  private boolean isPolicyOrderUpdate(PolicyDeploymentStatus status, PolicyDefinition policyDefinition) {
    return status.getAppliedPolicyStatus().isPresent() &&
        !sameOrder(status.getAppliedPolicyStatus().get().getPolicyDefinition(), policyDefinition) &&
        sameNonOrderConfiguration(status.getAppliedPolicyStatus().get().getPolicyDefinition(), policyDefinition);
  }

  private boolean isPolicyRevert(PolicyDeploymentStatus status, PolicyDefinition policyDefinition) {
    return status.getAppliedPolicyStatus().isPresent() &&
        !areEqual(status.getLatestPolicyStatus().getPolicyDefinition(), policyDefinition) &&
        areEqual(status.getAppliedPolicyStatus().get().getPolicyDefinition(), policyDefinition);
  }

  private boolean isPolicyUpdate(PolicyDeploymentStatus status, PolicyDefinition policyDefinition) {
    return !areEqual(status.getLatestPolicyStatus().getPolicyDefinition(), policyDefinition);
  }

  private boolean areEqual(PolicyDefinition oldDefinition, PolicyDefinition newDefinition) {
    PolicyDefinitionBuilder builder = new PolicyDefinitionBuilder(newDefinition);
    ignoreIdentityManagementIfNecessary(oldDefinition, builder);
    return oldDefinition.equals(builder.build());
  }

  private boolean sameOrder(PolicyDefinition oldDefinition, PolicyDefinition policyDefinition) {
    return oldDefinition.getOrder().equals(policyDefinition.getOrder());
  }

  private boolean sameNonOrderConfiguration(PolicyDefinition oldDefinition, PolicyDefinition policyDefinition) {
    PolicyDefinitionBuilder builder = new PolicyDefinitionBuilder(policyDefinition).order(oldDefinition.getOrder());
    ignoreIdentityManagementIfNecessary(oldDefinition, builder);
    return oldDefinition.equals(builder.build());
  }

  private void ignoreIdentityManagementIfNecessary(PolicyDefinition latestDefinition, PolicyDefinitionBuilder builder) {
    // Change in the identity management properties shouldn't be considered as a change unless they are present
    // in the already deployed policy.
    if (!hasIdentityManagementProperties(latestDefinition)) {
      builder.noIdentityManagement();
    }
  }

  private ChangeType ifApplied(PolicyDeploymentStatus status, PolicyDefinition newDefinition,
                               BiFunction<PolicyDefinition, PolicyDefinition, ChangeType> ifFunction,
                               Function<PolicyDefinition, ChangeType> elseFunction) {
    Optional<PolicyDefinitionDeploymentStatus> appliedStatus = status.getAppliedPolicyStatus();
    if (appliedStatus.isPresent()) {
      return ifFunction.apply(appliedStatus.get().getPolicyDefinition(), newDefinition);
    } else {
      return elseFunction.apply(newDefinition);
    }
  }
}
