/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.deployment;

import static java.util.Optional.empty;
import static java.util.Optional.of;

import com.mulesoft.mule.runtime.gw.logging.GatewayMuleAppLoggerFactory;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.PolicyDefinitionDeploymentStatus;
import com.mulesoft.mule.runtime.gw.policies.PolicyDeploymentStatus;
import com.mulesoft.mule.runtime.gw.policies.service.PolicyDeploymentTracker;
import com.mulesoft.mule.runtime.gw.policies.store.PolicyStore;

import org.slf4j.Logger;

public class DefaultTransactionalPolicyDeployer implements TransactionalPolicyDeployer {

  private static final Logger LOGGER = GatewayMuleAppLoggerFactory.getLogger(DefaultTransactionalPolicyDeployer.class);

  private final PolicyDeploymentTracker policyDeploymentTracker;
  private final PolicyStore policyStore;
  private final PolicyDeployer policyDeployer;

  public DefaultTransactionalPolicyDeployer(PolicyDeploymentTracker policyDeploymentTracker, PolicyStore policyStore,
                                            PolicyDeployer policyDeployer) {
    this.policyDeployer = policyDeployer;
    this.policyDeploymentTracker = policyDeploymentTracker;
    this.policyStore = policyStore;
  }

  @Override
  public PolicyDefinitionDeploymentStatus deploy(PolicyDefinition policyDefinition, Api api) {
    PolicyDefinitionDeploymentStatus status = policyDeployer.deploy(policyDefinition, api);
    if (status.isDeploymentSuccess()) {
      this.informSuccessfullyDeploy(status, api);
    } else {
      this.informDeploymentFailure(new PolicyDeploymentStatus(status, empty()), api);
    }
    return status;
  }

  @Override
  public PolicyDefinitionDeploymentStatus updateOrder(PolicyDefinition policy, Api api) {
    return policyDeployer.updateOrder(policy, api);
  }

  @Override
  public void updateOrder(PolicyDefinition oldPolicy, PolicyDefinition updatedPolicy, Api api) {
    PolicyDefinitionDeploymentStatus status = this.updateOrder(updatedPolicy, api);
    if (status.isDeploymentSuccess()) {
      this.informSuccessfullyDeploy(status, api);
    } else {
      LOGGER.info("Hot Reorder Failed for policy {} in app {}, unapplying and reapplying policy to change the order.",
                  updatedPolicy.getName(), api.getImplementation().getArtifactName());
      this.update(oldPolicy, updatedPolicy, api);
    }
  }

  @Override
  public void update(PolicyDefinition oldPolicy, PolicyDefinition updatedPolicy, Api api) {
    undeploy(oldPolicy.getName(), api);
    PolicyDefinitionDeploymentStatus status = policyDeployer.deploy(updatedPolicy, api);
    if (status.isDeploymentSuccess()) {
      this.informSuccessfullyDeploy(status, api);
    } else {
      undeploy(updatedPolicy.getName(), api);
      PolicyDefinitionDeploymentStatus revertStatus = policyDeployer.deploy(oldPolicy, api);
      if (revertStatus.isDeploymentSuccess()) {
        LOGGER.info("Successfully reverted {} in app {} to previous configuration.", oldPolicy.getName(),
                    api.getImplementation().getArtifactName());
        informDeploymentFailure(new PolicyDeploymentStatus(status, of(revertStatus)), api);
      } else {
        LOGGER.error("Failed to revert {} in app {} to previous configuration, api will remain unprotected.", oldPolicy.getName(),
                     api.getImplementation().getArtifactName());
        informDeploymentFailure(new PolicyDeploymentStatus(status, empty()), api);
      }
    }
  }

  @Override
  public void revertPolicy(PolicyDefinition policy, Api api) {
    this.informSuccessfullyDeploy(new PolicyDefinitionDeploymentStatus(policy), api);
  }

  @Override
  public boolean undeploy(String policyName, Api api) {
    boolean result = policyDeployer.undeploy(policyName, api);
    policyDeploymentTracker.policyRemoved(api.getKey(), policyName);
    if (!policyDeploymentTracker.hasDeployments(policyName)) {
      policyStore.remove(policyName);
    }
    return result;
  }

  private void informSuccessfullyDeploy(PolicyDefinitionDeploymentStatus status, Api api) {
    policyDeploymentTracker.policyRemoved(api.getImplementation().getApiKey(), status.getPolicyDefinition().getName());
    policyDeploymentTracker.policyDeployed(api.getImplementation().getApiKey(), new PolicyDeploymentStatus(status));
    policyStore.cleanDeploymentFailure(status.getPolicyDefinition(), api.getKey());
  }

  private void informDeploymentFailure(PolicyDeploymentStatus status, Api api) {
    policyDeploymentTracker.policyRemoved(api.getImplementation().getApiKey(),
                                          status.getLatestPolicyStatus().getPolicyDefinition().getName());
    policyDeploymentTracker.policyDeployed(api.getImplementation().getApiKey(), status);
    policyStore.store(status.getAppliedPolicyStatus().orElseGet(status::getLatestPolicyStatus).getPolicyDefinition());
    policyStore.storeDeploymentFailure(status.getLatestPolicyStatus().getPolicyDefinition(), api.getKey(),
                                       status.getLatestPolicyStatus().getException());
  }


}
