/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.factory;

import static com.google.common.collect.Lists.newArrayList;
import static com.mulesoft.mule.runtime.gw.policies.pointcut.CompositePointcut.and;
import static com.mulesoft.mule.runtime.gw.policies.pointcut.CompositePointcut.or;
import static java.lang.Boolean.TRUE;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;

import org.mule.runtime.api.notification.NotificationListener;
import org.mule.runtime.core.api.policy.PolicyParametrization;
import org.mule.runtime.policy.api.PolicyPointcut;

import com.mulesoft.mule.runtime.gw.hdp.config.HighDensityProxyConfiguration;
import com.mulesoft.mule.runtime.gw.model.ApiImplementation;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.pointcut.ApiPointcutAdapter;
import com.mulesoft.mule.runtime.gw.policies.pointcut.CompositePointcut;
import com.mulesoft.mule.runtime.gw.policies.pointcut.EffectiveHttpResourcePointcut;
import com.mulesoft.mule.runtime.gw.policies.pointcut.FullPathExtractor;
import com.mulesoft.mule.runtime.gw.policies.pointcut.HttpHeaderPointcut;
import com.mulesoft.mule.runtime.gw.policies.pointcut.IsHttpComponentPointcut;
import com.mulesoft.mule.runtime.gw.policies.pointcut.MaskedPathExtractor;
import com.mulesoft.mule.runtime.gw.policies.pointcut.PolicyPointcutParametersPathExtractor;

import com.google.common.collect.ImmutableMap;

import java.io.File;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Creates the policy parametrization for a API implementation from a policy definition
 */
public class PolicyParametrizationFactory {

  private static final Logger LOGGER = LoggerFactory.getLogger(PolicyParametrizationFactory.class);

  private static final String PARAMETER_ENCRYPTED_PROPERTIES_FILE = "encryptedPropertiesFile";

  private static final String API_ID = "apiId";
  private static final String IS_WSDL_ENDPOINT = "isWsdlEndpoint";
  private static final String IS_POLICY_REORDER = "isPolicyReorder";
  private static final String IS_WSDL_ENDPOINT_DEFAULT = "false";

  private final HighDensityProxyConfiguration hdpConfiguration = new HighDensityProxyConfiguration();

  /**
   * Creates a {@link PolicyParametrization} for the specified API implementation, using the policy definition.
   *
   * @param policyDefinition the definition to use to create the parametrization
   * @param implementation the API implementation for which the parametrization is for
   * @param policyConfigFile the config file of the policy, needed for the parametrization
   * @param encryptedPropertiesFile the YAML file containing the encrypted configuration, if enabled
   * @param isPolicyReorder indicates if it is a parametrization for a policy reorder
   * @param notificationListeners listeners to add to receive notifications sent by the policy's context
   * @return the policy parametrization
   */
  public PolicyParametrization create(PolicyDefinition policyDefinition, ApiImplementation implementation,
                                      File policyConfigFile, File encryptedPropertiesFile, Boolean isPolicyReorder,
                                      List<NotificationListener> notificationListeners) {

    Map<String, String> defaultParameters =
        createDefaultParameters(implementation, policyDefinition, encryptedPropertiesFile, isPolicyReorder);

    PolicyPointcut effectivePointcut = createEffectivePointcut(policyDefinition, implementation);

    String name = buildParametrizationId(policyDefinition.getName(), implementation);

    return new PolicyParametrization(name,
                                     effectivePointcut,
                                     policyDefinition.getOrder(),
                                     defaultParameters,
                                     policyConfigFile,
                                     notificationListeners);
  }

  /**
   * Builds the parametrization ID that is going to use for the policy deployment. It consists of the policy name and the
   * implementation flow name. The flow name is used to avoid ID collisions when the policy has to be deployed to multiple APIs
   * that live in the same Application.
   *
   * @param policyName name of the policy
   * @param implementation API implementation
   * @return the parametrization's ID
   */
  public String buildParametrizationId(String policyName, ApiImplementation implementation) {
    return policyName + "-" + implementation.getFlow().getName();
  }

  private Map<String, String> createDefaultParameters(ApiImplementation implementation, PolicyDefinition policyDefinition,
                                                      File encryptedPropertiesFile, boolean isPolicyReorder) {
    policyDefinition.getConfigurationData().getConfiguration().getOrDefault(IS_WSDL_ENDPOINT, IS_WSDL_ENDPOINT_DEFAULT);

    String wsdlConfiguration = policyDefinition.getConfigurationData()
        .getConfiguration()
        .getOrDefault(IS_WSDL_ENDPOINT, IS_WSDL_ENDPOINT_DEFAULT).toString();

    ImmutableMap.Builder<String, String> builder = new ImmutableMap.Builder<String, String>()
        .putAll(getEncryptedProperties(encryptedPropertiesFile))
        .put(API_ID, implementation.getApiKey().id().toString())
        .put(IS_WSDL_ENDPOINT, wsdlConfiguration);

    if (isPolicyReorder) {
      builder.put(IS_POLICY_REORDER, TRUE.toString());
    }

    return builder.build();
  }

  private PolicyPointcut createEffectivePointcut(PolicyDefinition policyDefinition, ApiImplementation implementation) {
    String flowName = implementation.getFlow().getName();
    ApiPointcutAdapter apiPointcutAdapter = new ApiPointcutAdapter(policyDefinition.getName(), flowName);

    CompositePointcut pointcut = and(apiPointcutAdapter, new IsHttpComponentPointcut(policyDefinition.getName()));
    if (implementation.isHdp()) {
      pointcut = applyHdpPointcut(policyDefinition.getName(), pointcut, implementation);
    }

    if (policyDefinition.getHttpResourcePointcuts().isEmpty()) {
      LOGGER.debug("There are no resource pointcuts defined for policy {}. It will be applied to flow {}",
                   policyDefinition.getName(), flowName);

      return pointcut;
    }

    PolicyPointcutParametersPathExtractor pathExtractor =
        implementation.isIgnoreBasePathOnResourceLevel() ? new MaskedPathExtractor() : new FullPathExtractor();

    List<EffectiveHttpResourcePointcut> resourcePointcuts = policyDefinition.getHttpResourcePointcuts().stream()
        .map(httpResourcePointcut -> new EffectiveHttpResourcePointcut(policyDefinition.getName(),
                                                                       httpResourcePointcut.getPath(),
                                                                       httpResourcePointcut.getMethod(),
                                                                       pathExtractor))
        .collect(toList());

    return and(pointcut, or(newArrayList(resourcePointcuts)));
  }

  private CompositePointcut applyHdpPointcut(String policyName, CompositePointcut pointcut, ApiImplementation implementation) {
    return implementation.getHdpService().map(service -> {
      HttpHeaderPointcut headerPointcut = new HttpHeaderPointcut(policyName, hdpConfiguration.getHdpServiceHeader(), service);
      LOGGER.debug("Injecting header pointcut {} for service {}", headerPointcut.toString(), service);
      return and(pointcut, headerPointcut);
    }).orElse(pointcut);
  }

  private Map<String, String> getEncryptedProperties(File encryptedProperties) {
    Map<String, String> parameters = new HashMap<>();

    ofNullable(encryptedProperties)
        .map(File::getAbsolutePath)
        .ifPresent((absolutePath) -> parameters.put(PARAMETER_ENCRYPTED_PROPERTIES_FILE, absolutePath));

    return parameters;
  }
}
