/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.service.detection;

import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.PolicyDefinitionDeploymentStatus;
import com.mulesoft.mule.runtime.gw.policies.PolicyDeploymentStatus;
import com.mulesoft.mule.runtime.gw.policies.service.PolicyDeploymentService;
import com.mulesoft.mule.runtime.gw.policies.service.detection.change.ChangeTypeVisitor;
import com.mulesoft.mule.runtime.gw.policies.service.detection.change.PolicyAdded;
import com.mulesoft.mule.runtime.gw.policies.service.detection.change.PolicyUpdate;
import com.mulesoft.mule.runtime.gw.policies.service.detection.change.PolicyRemoved;
import com.mulesoft.mule.runtime.gw.policies.service.detection.change.PolicyReorder;
import com.mulesoft.mule.runtime.gw.policies.service.detection.change.PolicyRevert;
import com.mulesoft.mule.runtime.gw.policies.service.detection.change.PolicyUnmodified;

import java.util.List;

/**
 * A class that can discern, given the current policy status and the incoming information of what policies should be applied which
 * actions should be taken for each policy
 */
public class PolicyChangeProcessor implements ChangeTypeVisitor {

  private final PolicySetChangeInspector inspector = new PolicySetChangeInspector();
  private final PolicyDeploymentService policyDeploymentService;

  public PolicyChangeProcessor(PolicyDeploymentService policyDeploymentService) {
    this.policyDeploymentService = policyDeploymentService;
  }

  /**
   * Given the current deployment statuses of all policies and the desired policy definitions takes action to consolidate the
   * current status with with the desired status.
   *
   * @param currentStatusList a {@link List} of {@link PolicyDefinitionDeploymentStatus} that reflects the current status.
   * @param newDefinitions a {@link List} of {@link PolicyDefinition} that should be currently applied.
   */
  public void process(List<PolicyDeploymentStatus> currentStatusList, List<PolicyDefinition> newDefinitions) {
    inspector.detectChanges(currentStatusList, newDefinitions)
        .forEach(change -> change.accept(this));
  }

  @Override
  public void visit(PolicyAdded policyAdded) {
    policyDeploymentService.newPolicy(policyAdded.getDefinition());
  }

  @Override
  public void visit(PolicyUpdate policyUpdate) {
    policyDeploymentService.updatePolicy(policyUpdate.getAppliedDefinition(),
                                         policyUpdate.getDefinition());
  }

  @Override
  public void visit(PolicyRemoved policyRemoved) {
    policyDeploymentService.removePolicy(policyRemoved.getDefinition());
  }

  @Override
  public void visit(PolicyReorder policyReorder) {
    policyDeploymentService.reorderPolicy(policyReorder.getAppliedDefinition(), policyReorder.getDefinition());
  }

  @Override
  public void visit(PolicyRevert policyRevert) {
    policyDeploymentService.revertPolicy(policyRevert.getDefinition());
  }

  @Override
  public void visit(PolicyUnmodified policyUnmodified) {}
}
