/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.deployment;

import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideLogger;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideVariable;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertThat;
import static org.mockito.Answers.RETURNS_DEEP_STUBS;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.reset;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.verifyZeroInteractions;
import static org.mockito.Mockito.when;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_HOME_DIRECTORY_PROPERTY;

import org.mule.runtime.core.api.construct.Flow;
import org.mule.runtime.core.api.policy.PolicyParametrization;
import org.mule.runtime.deployment.model.api.application.Application;
import org.mule.runtime.deployment.model.api.application.ApplicationPolicyManager;
import org.mule.runtime.deployment.model.api.application.ApplicationStatus;
import org.mule.runtime.deployment.model.api.policy.PolicyRegistrationException;
import org.mule.runtime.deployment.model.api.policy.PolicyTemplateDescriptor;
import org.mule.runtime.policy.api.PolicyPointcut;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.SystemPropertyTemporaryFolder;

import com.mulesoft.anypoint.tests.logger.DebugLine;
import com.mulesoft.anypoint.tests.logger.InfoLine;
import com.mulesoft.anypoint.tests.logger.MockLogger;
import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.ApiImplementation;
import com.mulesoft.mule.runtime.gw.model.EmptyPolicySpecification;
import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.Policy;
import com.mulesoft.mule.runtime.gw.policies.PolicyDefinitionDeploymentStatus;
import com.mulesoft.mule.runtime.gw.policies.PolicyDeploymentStatus;
import com.mulesoft.mule.runtime.gw.policies.factory.PolicyFactory;
import com.mulesoft.mule.runtime.gw.policies.factory.PolicyParametrizationFactory;
import com.mulesoft.mule.runtime.gw.policies.notification.PolicyNotificationListenerSuppliers;
import com.mulesoft.mule.runtime.gw.policies.service.DefaultPolicyDeploymentTracker;
import com.mulesoft.mule.runtime.gw.policies.service.PolicyDeploymentListener;
import com.mulesoft.mule.runtime.gw.policies.store.PolicyStore;
import com.mulesoft.mule.runtime.gw.policies.template.PolicyTemplate;
import com.mulesoft.mule.runtime.gw.policies.template.exception.PolicyTemplateResolverException;

import java.io.File;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class DefaultPolicyDeployerTestCase extends AbstractMuleTestCase {

  private static final String FLOW_NAME = "flowName";
  private static final String RESOLVED_TEMPLATE = "resolvedTemplate";

  @Rule
  public SystemPropertyTemporaryFolder temporaryFolder = new SystemPropertyTemporaryFolder(MULE_HOME_DIRECTORY_PROPERTY);

  @Mock
  private Api api;
  @Mock(answer = RETURNS_DEEP_STUBS)
  private Application application;
  @Mock
  private ApplicationPolicyManager policyManager;
  @Mock
  private PolicyParametrizationFactory policyParametrizationFactory;
  @Mock
  private PolicyStore policyStore;


  private PolicyTemplateDescriptor templateDescriptor;
  private PolicyParametrization policyParametrization;

  private Policy policy;
  private PolicyDefinition policyDefinition;

  private DefaultPolicyDeploymentTracker policyDeploymentTracker;
  private PolicyDeploymentListener policyDeploymentListener;

  private ApiImplementation apiImplementation;

  private TransactionalPolicyDeployer transactionalPolicyDeployer;
  private MockLogger logger;
  private PolicyFactory policyFactory;

  @Before
  public void setUp() throws PolicyTemplateResolverException {
    PolicyTemplate template = mock(PolicyTemplate.class);
    Flow flow = mock(Flow.class);

    logger = new MockLogger();
    policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, emptyList(), 1, new PolicyConfiguration(emptyMap()));

    policyFactory = mock(PolicyFactory.class);
    policy = new Policy(template, policyDefinition, RESOLVED_TEMPLATE);
    when(policyFactory.createFromPolicyDefinition(policyDefinition)).thenReturn(policy);

    apiImplementation = new ApiImplementation(API_KEY, application, flow, false);

    policyDeploymentTracker = new DefaultPolicyDeploymentTracker();
    policyDeploymentListener = mock(PolicyDeploymentListener.class);
    policyDeploymentTracker.addPolicyDeploymentListener(policyDeploymentListener);

    PolicyDeployer defaultPolicyDeployer = new DefaultPolicyDeployer(policyStore, policyFactory,
                                                                     new PolicyNotificationListenerSuppliers());
    transactionalPolicyDeployer =
        new DefaultTransactionalPolicyDeployer(policyDeploymentTracker, policyStore, defaultPolicyDeployer);

    policyParametrization =
        new PolicyParametrization("id", mock(PolicyPointcut.class), 1, emptyMap(), mock(File.class), emptyList());
    templateDescriptor = new PolicyTemplateDescriptor("name");

    when(api.getImplementation()).thenReturn(apiImplementation);
    when(api.getKey()).thenReturn(API_KEY);
    when(flow.getName()).thenReturn(FLOW_NAME);
    when(application.getStatus()).thenReturn(ApplicationStatus.STARTED);
    when(application.getPolicyManager()).thenReturn(policyManager);
    when(policyParametrizationFactory.create(any(), any(), any(), any(), any(), any())).thenReturn(policyParametrization);
    when(template.getTemplateDescriptor()).thenReturn(templateDescriptor);
    when(template.getPolicySpecification()).thenReturn(new EmptyPolicySpecification());
    when(application.getArtifactClassLoader().getClassLoader()).thenReturn(Thread.currentThread().getContextClassLoader());

    overrideVariable("policyDeployer.policyParametrizationFactory").in(
                                                                       transactionalPolicyDeployer)
        .with(policyParametrizationFactory);
    overrideLogger().in(defaultPolicyDeployer).with(logger);
    overrideLogger().in(transactionalPolicyDeployer).with(logger);
  }

  @Test
  public void deployPolicy() throws PolicyRegistrationException {
    transactionalPolicyDeployer.deploy(policyDefinition, api);

    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY), hasSize(1));
    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).get(0).getLatestPolicyStatus().isDeploymentSuccess(),
               is(true));
    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).get(0).getLatestPolicyStatus().isTemplateDownloadFailed(),
               is(false));
    verify(policyManager).addPolicy(templateDescriptor, policyParametrization);
    verify(policyStore).store(policy);
    verify(policyStore).cleanDeploymentFailure(policyDefinition, API_KEY);
    assertThat("Log lines size does not match", logger.lines(), hasSize(2));
    assertThat(logger.lines().get(0),
               is(new DebugLine("Applying policy {} version {} to {} in application {}",
                                policyDefinition.getName(),
                                policyDefinition.getTemplateKey().getVersion(),
                                api,
                                api.getImplementation().getArtifactName())));
    assertThat(logger.lines().get(1),
               is(new InfoLine("Applied policy {} version {} to {} in application {}",
                               policyDefinition.getName(),
                               policyDefinition.getTemplateKey().getVersion(),
                               api,
                               api.getImplementation().getArtifactName())));

    verify(policyDeploymentListener).policyDeployed(API_KEY, successfulStatus(policyDefinition));
    verifyNoMoreInteractions(policyDeploymentListener);
  }

  @Test
  public void deployPolicyFails() throws PolicyRegistrationException {
    PolicyRegistrationException exception = new PolicyRegistrationException(null, null);
    doThrow(exception).when(policyManager).addPolicy(any(), any());

    transactionalPolicyDeployer.deploy(policyDefinition, api);

    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY), hasSize(1));
    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).get(0).getLatestPolicyStatus().isDeploymentSuccess(),
               is(false));
    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).get(0).getLatestPolicyStatus().isTemplateDownloadFailed(),
               is(false));
    verify(policyManager).addPolicy(templateDescriptor, policyParametrization);
    verify(policyStore).store(policy);
    verify(policyStore).storeDeploymentFailure(policyDefinition, apiImplementation.getApiKey(), exception);
  }

  @Test
  public void undeployPolicy() {
    PolicyDeploymentStatus status = successfulStatus(policyDefinition);
    policyDeploymentTracker.policyDeployed(API_KEY, status);
    reset(policyDeploymentListener); // Previous call added the policy for the listener

    when(policyManager.removePolicy(any())).thenReturn(true);

    boolean undeployed = transactionalPolicyDeployer.undeploy(policyDefinition.getName(), api);

    assertThat(undeployed, is(true));
    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY), empty());
    verify(policyStore).remove(policyDefinition.getName());

    verify(policyDeploymentListener).policyRemoved(API_KEY, status);
    verifyNoMoreInteractions(policyDeploymentListener);
  }

  @Test
  public void undeployPolicyWithMultipleDeployments() {
    PolicyDeploymentStatus status = successfulStatus(policyDefinition);
    policyDeploymentTracker.policyDeployed(API_KEY, status);
    policyDeploymentTracker.policyDeployed(API_KEY_2, status);
    reset(policyDeploymentListener); // Previous call added the policy for the listener
    when(policyManager.removePolicy(any())).thenReturn(true);

    boolean undeployed = transactionalPolicyDeployer.undeploy(policyDefinition.getName(), api);

    assertThat(undeployed, is(true));
    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY), empty());
    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY_2), hasSize(1));
    verifyZeroInteractions(policyStore);

    verify(policyDeploymentListener).policyRemoved(API_KEY, status);
    verifyNoMoreInteractions(policyDeploymentListener);
  }

  @Test
  public void undeployNotDeployedPolicy() {
    when(policyManager.removePolicy(policyDefinition.getName())).thenReturn(false);

    boolean undeployed = transactionalPolicyDeployer.undeploy(policyDefinition.getName(), api);

    assertThat(undeployed, is(false));
    verify(policyStore).remove(policyDefinition.getName());
    verifyNoMoreInteractions(policyDeploymentListener);
  }

  private PolicyDeploymentStatus successfulStatus(PolicyDefinition definition) {
    return new PolicyDeploymentStatus(new PolicyDefinitionDeploymentStatus(definition));
  }

}
