/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.pointcut;

import static com.mulesoft.mule.runtime.gw.api.policy.HttpResourcePointcut.HTTP_LISTENER_COMPONENT_IDENTIFIER;
import static com.mulesoft.mule.runtime.gw.api.policy.HttpResourcePointcut.HTTP_REQUEST_COMPONENT_IDENTIFIER;
import static com.mulesoft.mule.runtime.gw.policies.pointcut.PointcutTestUtils.annotatedListenerPointcutParameters;
import static com.mulesoft.mule.runtime.gw.policies.pointcut.PointcutTestUtils.annotatedRequesterPointcutParameters;
import static com.mulesoft.mule.runtime.gw.policies.pointcut.PointcutTestUtils.listenerPointcutParameters;
import static com.mulesoft.mule.runtime.gw.policies.pointcut.PointcutTestUtils.nonHttpPointcutParameters;
import static com.mulesoft.mule.runtime.gw.policies.pointcut.PointcutTestUtils.requesterPointcutParameters;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideLogger;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideVariable;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.policy.api.PolicyPointcutParameters;
import org.mule.tck.junit4.AbstractMuleTestCase;

import com.mulesoft.anypoint.tests.logger.MockLogger;
import com.mulesoft.anypoint.tests.logger.TraceLine;

import org.junit.Before;
import org.junit.Test;

public class IsHttpComponentPointcutTestCase extends AbstractMuleTestCase {

  private static final String POLICY_NAME = "name";

  private static final ComponentIdentifier INVALID_COMPONENT_IDENTIFIER =
      ComponentIdentifier.builder().namespace("invalid").name("component").build();

  private MockLogger logger;

  @Before
  public void setUp() {
    logger = new MockLogger();
  }

  @Test
  public void httpListenerComponentIsValid() {
    IsHttpComponentPointcut pointcut = new IsHttpComponentPointcut(POLICY_NAME);
    overrideLogger().in(pointcut).with(logger);
    PolicyPointcutParameters parameters = listenerPointcutParameters("");

    boolean matches = pointcut.matches(parameters);

    assertTrue(matches);
    assertThat(logger.lines(), hasSize(1));
    assertThat(logger.lines().get(0),
               is(new TraceLine("Request matches pointcut of policy {} since {} is a valid HTTP component.", POLICY_NAME,
                                HTTP_LISTENER_COMPONENT_IDENTIFIER)));
  }

  @Test
  public void httpRequesterComponentIsValid() {
    IsHttpComponentPointcut pointcut = new IsHttpComponentPointcut(POLICY_NAME);
    overrideLogger().in(pointcut).with(logger);
    PolicyPointcutParameters parameters = requesterPointcutParameters("");

    boolean matches = pointcut.matches(parameters);

    assertTrue(matches);
    assertThat(logger.lines(), hasSize(1));
    assertThat(logger.lines().get(0),
               is(new TraceLine("Request matches pointcut of policy {} since {} is a valid HTTP component.",
                                POLICY_NAME, HTTP_REQUEST_COMPONENT_IDENTIFIER)));
  }

  @Test
  public void annotatedHttpListenerComponentIsValid() {
    IsHttpComponentPointcut pointcut = new IsHttpComponentPointcut(POLICY_NAME);
    overrideLogger().in(pointcut).with(logger);
    PolicyPointcutParameters parameters = annotatedListenerPointcutParameters("");

    boolean matches = pointcut.matches(parameters);

    assertTrue(matches);
    assertThat(logger.lines(), hasSize(2));
    assertThat(logger.lines().get(0),
               is(new TraceLine("Annotation 'api-gateway:disablePolicies' has no effect on component {}",
                                HTTP_LISTENER_COMPONENT_IDENTIFIER)));
    assertThat(logger.lines().get(1),
               is(new TraceLine("Request matches pointcut of policy {} since {} is a valid HTTP component.",
                                POLICY_NAME, HTTP_LISTENER_COMPONENT_IDENTIFIER)));
  }

  @Test
  public void annotatedHttpRequesterComponentIsNotValid() {
    IsHttpComponentPointcut pointcut = new IsHttpComponentPointcut(POLICY_NAME);
    overrideLogger().in(pointcut).with(logger);
    overrideVariable("notApplicableWarn.logger").in(pointcut).with(logger);
    PolicyPointcutParameters parameters = annotatedRequesterPointcutParameters("");

    boolean matches = pointcut.matches(parameters);

    assertFalse(matches);
    assertThat(logger.lines(), hasSize(3));
    assertTrue(logger.lines().get(0).toString()
        .contains("is annotated with 'api-gateway:disablePolicies', no operation block policies will be applied to this element"));
    assertTrue(logger.lines().get(1).toString()
        .contains("does not apply for pointcut for being annotated with 'api-gateway:disablePolicies'"));
    assertThat(logger.lines().get(2),
               is(new TraceLine("Request does not match pointcut of policy {} since {} is not a valid HTTP component.",
                                POLICY_NAME, HTTP_REQUEST_COMPONENT_IDENTIFIER)));
  }

  @Test
  public void multipleCallToAnnotatedHttpRequesterWarnsOnce() {
    IsHttpComponentPointcut pointcut = new IsHttpComponentPointcut(POLICY_NAME);
    overrideLogger().in(pointcut).with(logger);
    overrideVariable("notApplicableWarn.logger").in(pointcut).with(logger);
    PolicyPointcutParameters parameters = annotatedRequesterPointcutParameters("");

    boolean matches = pointcut.matches(parameters);

    assertFalse(matches);
    assertThat(logger.lines(), hasSize(3));
    assertTrue(logger.lines().get(0).toString()
        .contains("is annotated with 'api-gateway:disablePolicies', no operation block policies will be applied to this element"));
    assertTrue(logger.lines().get(1).toString()
        .contains("does not apply for pointcut for being annotated with 'api-gateway:disablePolicies'"));
    assertThat(logger.lines().get(2),
               is(new TraceLine("Request does not match pointcut of policy {} since {} is not a valid HTTP component.",
                                POLICY_NAME, HTTP_REQUEST_COMPONENT_IDENTIFIER)));

    logger.lines().clear();
    matches = pointcut.matches(parameters);

    assertFalse(matches);
    assertThat(logger.lines(), hasSize(2));
    assertTrue(logger.lines().get(0).toString()
        .contains("does not apply for pointcut for being annotated with 'api-gateway:disablePolicies'"));
    assertThat(logger.lines().get(1),
               is(new TraceLine("Request does not match pointcut of policy {} since {} is not a valid HTTP component.",
                                POLICY_NAME, HTTP_REQUEST_COMPONENT_IDENTIFIER)));
  }

  @Test
  public void anotherComponentIsNotValid() {
    IsHttpComponentPointcut pointcut = new IsHttpComponentPointcut(POLICY_NAME);
    overrideLogger().in(pointcut).with(logger);
    PolicyPointcutParameters parameters = nonHttpPointcutParameters("");

    boolean matches = pointcut.matches(parameters);

    assertFalse(matches);
    assertThat(logger.lines(), hasSize(1));
    assertThat(logger.lines().get(0),
               is(new TraceLine("Request does not match pointcut of policy {} since {} is not a valid HTTP component.",
                                POLICY_NAME, INVALID_COMPONENT_IDENTIFIER)));
  }

}
