/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.service;

import static com.google.common.collect.Lists.newArrayList;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static com.mulesoft.mule.runtime.gw.policies.service.DeploymentStatusTestFactory.applied;
import static com.mulesoft.mule.runtime.gw.policies.service.DeploymentStatusTestFactory.appliedDefinition;
import static com.mulesoft.mule.runtime.gw.policies.service.DeploymentStatusTestFactory.deploymentFailed;
import static com.mulesoft.mule.runtime.gw.policies.service.DeploymentStatusTestFactory.deploymentFailedDefinition;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonMap;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;
import static org.mockito.Mockito.inOrder;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.tck.junit4.AbstractMuleTestCase;

import com.mulesoft.mule.runtime.gw.model.Api;
import com.mulesoft.mule.runtime.gw.model.ApiImplementation;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.deployment.DefaultTransactionalPolicyDeployer;
import com.mulesoft.mule.runtime.gw.policies.deployment.PolicyDeployer;
import com.mulesoft.mule.runtime.gw.policies.deployment.TransactionalPolicyDeployer;
import com.mulesoft.mule.runtime.gw.policies.store.PolicyStore;

import org.junit.Before;
import org.junit.Test;
import org.mockito.InOrder;

public class TransactionalPolicyDeployerTestCase extends AbstractMuleTestCase {

  private static final PolicyDefinition definition =
      new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, newArrayList(), 1, emptyMap());
  private static final PolicyDefinition differentDefinition =
      new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, newArrayList(), 1, singletonMap("param", "value"));

  private static final String definitionName = definition.getName();

  private Api api;
  private PolicyStore policyStore;
  private PolicyDeploymentTracker policyDeploymentTracker;
  private PolicyDeployer policyDeployer;
  private TransactionalPolicyDeployer transactionalDeployer;

  @Before
  public void setUp() {
    api = mockAPi();
    policyDeploymentTracker = new DefaultPolicyDeploymentTracker();
    policyStore = mock(PolicyStore.class);
    policyDeployer = mock(PolicyDeployer.class);


    transactionalDeployer = new DefaultTransactionalPolicyDeployer(policyDeploymentTracker,
                                                                   policyStore,
                                                                   policyDeployer);
  }

  @Test
  public void reorderSuccess() {
    policyDeploymentTracker.policyDeployed(API_KEY, applied(differentDefinition));
    when(policyDeployer.updateOrder(definition, api)).thenReturn(appliedDefinition(definition));

    transactionalDeployer.updateOrder(differentDefinition, definition, api);

    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).size(), is(1));
    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).get(0), is(applied(definition)));

    InOrder inOrder = inOrder(policyDeployer, policyStore);
    inOrder.verify(policyDeployer).updateOrder(definition, api);
    inOrder.verify(policyStore).cleanDeploymentFailure(definition, API_KEY);
    inOrder.verifyNoMoreInteractions();
  }

  @Test
  public void hotReorderFails() {
    policyDeploymentTracker.policyDeployed(API_KEY, applied(differentDefinition));
    when(policyDeployer.updateOrder(definition, api)).thenReturn(deploymentFailedDefinition(definition));
    when(policyDeployer.undeploy(definitionName, api)).thenReturn(true);
    when(policyDeployer.deploy(definition, api)).thenReturn(appliedDefinition(definition));

    transactionalDeployer.updateOrder(differentDefinition, definition, api);

    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).size(), is(1));
    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).get(0), is(applied(definition)));

    InOrder inOrder = inOrder(policyDeployer, policyStore);
    inOrder.verify(policyDeployer).updateOrder(definition, api);
    inOrder.verify(policyDeployer).undeploy(definitionName, api);
    inOrder.verify(policyStore).remove(definitionName);
    inOrder.verify(policyDeployer).deploy(definition, api);
    inOrder.verify(policyStore).cleanDeploymentFailure(definition, API_KEY);
    inOrder.verifyNoMoreInteractions();
  }

  @Test
  public void hotReorderFailsUpdateFails() {
    policyDeploymentTracker.policyDeployed(API_KEY, applied(differentDefinition));
    when(policyDeployer.updateOrder(definition, api)).thenReturn(deploymentFailedDefinition(definition));
    when(policyDeployer.undeploy(definitionName, api)).thenReturn(true);
    when(policyDeployer.deploy(definition, api)).thenReturn(deploymentFailedDefinition(definition));
    when(policyDeployer.deploy(differentDefinition, api)).thenReturn(appliedDefinition(differentDefinition));

    transactionalDeployer.updateOrder(differentDefinition, definition, api);

    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).size(), is(1));
    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).get(0),
               is(deploymentFailed(differentDefinition, definition)));

    InOrder inOrder = inOrder(policyDeployer, policyStore);
    inOrder.verify(policyDeployer).updateOrder(definition, api);
    inOrder.verify(policyDeployer).undeploy(definitionName, api);
    inOrder.verify(policyStore).remove(definitionName);
    inOrder.verify(policyDeployer).deploy(definition, api);
    inOrder.verify(policyStore).remove(definitionName);
    inOrder.verify(policyDeployer).deploy(differentDefinition, api);
    inOrder.verify(policyStore).store(differentDefinition);
    inOrder.verify(policyStore).storeDeploymentFailure(definition, API_KEY, null);
    inOrder.verifyNoMoreInteractions();
  }

  @Test
  public void hotReorderFailsUpdateFailsRevertFails() {
    policyDeploymentTracker.policyDeployed(API_KEY, applied(differentDefinition));
    when(policyDeployer.updateOrder(definition, api)).thenReturn(deploymentFailedDefinition(definition));
    when(policyDeployer.undeploy(definitionName, api)).thenReturn(true);
    when(policyDeployer.deploy(definition, api)).thenReturn(deploymentFailedDefinition(definition));
    when(policyDeployer.deploy(differentDefinition, api)).thenReturn(deploymentFailedDefinition(differentDefinition));

    transactionalDeployer.updateOrder(differentDefinition, definition, api);

    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).size(), is(1));
    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).get(0), is(deploymentFailed(definition)));

    InOrder inOrder = inOrder(policyDeployer, policyStore);
    inOrder.verify(policyDeployer).updateOrder(definition, api);
    inOrder.verify(policyDeployer).undeploy(definitionName, api);
    inOrder.verify(policyStore).remove(definitionName);
    inOrder.verify(policyDeployer).deploy(definition, api);
    inOrder.verify(policyStore).remove(definitionName);
    inOrder.verify(policyDeployer).deploy(differentDefinition, api);
    inOrder.verify(policyStore).store(definition);
    inOrder.verify(policyStore).storeDeploymentFailure(definition, API_KEY, null);
    inOrder.verifyNoMoreInteractions();
  }

  @Test
  public void updateSuccess() {
    policyDeploymentTracker.policyDeployed(API_KEY, applied(differentDefinition));
    when(policyDeployer.undeploy(definitionName, api)).thenReturn(true);
    when(policyDeployer.deploy(definition, api)).thenReturn(appliedDefinition(definition));

    transactionalDeployer.update(differentDefinition, definition, api);

    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).size(), is(1));
    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).get(0), is(applied(definition)));

    InOrder inOrder = inOrder(policyDeployer, policyStore);
    inOrder.verify(policyDeployer).undeploy(definitionName, api);
    inOrder.verify(policyStore).remove(definitionName);
    inOrder.verify(policyDeployer).deploy(definition, api);
    inOrder.verify(policyStore).cleanDeploymentFailure(definition, API_KEY);
    inOrder.verifyNoMoreInteractions();
  }

  @Test
  public void updateFails() {
    policyDeploymentTracker.policyDeployed(API_KEY, applied(differentDefinition));
    when(policyDeployer.undeploy(definitionName, api)).thenReturn(true);
    when(policyDeployer.deploy(definition, api)).thenReturn(deploymentFailedDefinition(definition));
    when(policyDeployer.deploy(differentDefinition, api)).thenReturn(appliedDefinition(differentDefinition));

    transactionalDeployer.update(differentDefinition, definition, api);

    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).size(), is(1));
    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).get(0),
               is(deploymentFailed(differentDefinition, definition)));

    InOrder inOrder = inOrder(policyDeployer, policyStore);
    inOrder.verify(policyDeployer).undeploy(definitionName, api);
    inOrder.verify(policyStore).remove(definitionName);
    inOrder.verify(policyDeployer).deploy(definition, api);
    inOrder.verify(policyStore).remove(definitionName);
    inOrder.verify(policyDeployer).deploy(differentDefinition, api);
    inOrder.verify(policyStore).store(differentDefinition);
    inOrder.verify(policyStore).storeDeploymentFailure(definition, API_KEY, null);
    inOrder.verifyNoMoreInteractions();
  }

  @Test
  public void updateFailsRevertFails() {
    policyDeploymentTracker.policyDeployed(API_KEY, applied(differentDefinition));
    when(policyDeployer.undeploy(definitionName, api)).thenReturn(true);
    when(policyDeployer.deploy(definition, api)).thenReturn(deploymentFailedDefinition(definition));
    when(policyDeployer.deploy(differentDefinition, api)).thenReturn(deploymentFailedDefinition(differentDefinition));

    transactionalDeployer.update(differentDefinition, definition, api);

    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).size(), is(1));
    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).get(0), is(deploymentFailed(definition)));

    InOrder inOrder = inOrder(policyDeployer, policyStore);
    inOrder.verify(policyDeployer).undeploy(definitionName, api);
    inOrder.verify(policyStore).remove(definitionName);
    inOrder.verify(policyDeployer).deploy(definition, api);
    inOrder.verify(policyStore).remove(definitionName);
    inOrder.verify(policyDeployer).deploy(differentDefinition, api);
    inOrder.verify(policyStore).store(definition);
    inOrder.verify(policyStore).storeDeploymentFailure(definition, API_KEY, null);
    inOrder.verifyNoMoreInteractions();
  }

  @Test
  public void updateWhenFailedDeploy() {
    policyDeploymentTracker.policyDeployed(API_KEY, deploymentFailed(differentDefinition));
    when(policyDeployer.undeploy(definitionName, api)).thenReturn(true);
    when(policyDeployer.deploy(definition, api)).thenReturn(appliedDefinition(definition));

    transactionalDeployer.deploy(definition, api);

    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).size(), is(1));
    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).get(0), is(applied(definition)));

    InOrder inOrder = inOrder(policyDeployer, policyStore);
    inOrder.verify(policyDeployer).deploy(definition, api);
    inOrder.verify(policyStore).cleanDeploymentFailure(definition, API_KEY);
    inOrder.verifyNoMoreInteractions();
  }

  @Test
  public void updateFailsWhenFailedDeploy() {
    policyDeploymentTracker.policyDeployed(API_KEY, deploymentFailed(differentDefinition));
    when(policyDeployer.undeploy(definitionName, api)).thenReturn(true);
    when(policyDeployer.deploy(definition, api)).thenReturn(deploymentFailedDefinition(definition));

    transactionalDeployer.deploy(definition, api);

    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).size(), is(1));
    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).get(0), is(deploymentFailed(definition)));

    InOrder inOrder = inOrder(policyDeployer, policyStore);
    inOrder.verify(policyDeployer).deploy(definition, api);
    inOrder.verify(policyStore).store(definition);
    inOrder.verify(policyStore).storeDeploymentFailure(definition, API_KEY, null);
    inOrder.verifyNoMoreInteractions();
  }

  @Test
  public void revertPolicy() {
    policyDeploymentTracker.policyDeployed(API_KEY, deploymentFailed(definition, differentDefinition));

    transactionalDeployer.revertPolicy(definition, api);

    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).size(), is(1));
    assertThat(policyDeploymentTracker.onlinePolicyStatuses(API_KEY).get(0), is(applied(definition)));

    InOrder inOrder = inOrder(policyDeployer, policyStore);
    inOrder.verify(policyStore).cleanDeploymentFailure(definition, API_KEY);
    inOrder.verifyNoMoreInteractions();
  }

  private Api mockAPi() {
    Api api = mock(Api.class);
    ApiImplementation impl = mock(ApiImplementation.class);
    when(api.getKey()).thenReturn(API_KEY);
    when(api.getImplementation()).thenReturn(impl);
    when(impl.getApiKey()).thenReturn(API_KEY);
    when(impl.getArtifactName()).thenReturn("MyArtifact");
    return api;
  }



}
