/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.template;

import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static com.mulesoft.mule.runtime.gw.api.PolicyFolders.getPolicyTemplatesFolder;
import static com.mulesoft.mule.runtime.gw.api.PolicyFolders.getPolicyTemplatesTempFolder;
import static com.mulesoft.mule.runtime.gw.policies.template.provider.FileSystemPolicyTemplateProvider.POLICY_TEMPLATE_LIGHT_PATTERN;
import static com.mulesoft.mule.runtime.gw.policies.template.provider.FileSystemPolicyTemplateProvider.POLICY_TEMPLATE_PATTERN;
import static com.mulesoft.mule.runtime.gw.policies.template.provider.FileSystemPolicyTemplateProvider.POLICY_YAML_PATTERN;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideLogger;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideVariable;
import static org.apache.commons.io.FileUtils.copyInputStreamToFile;
import static org.apache.commons.io.FileUtils.openInputStream;
import static org.hamcrest.Matchers.arrayWithSize;
import static org.hamcrest.Matchers.instanceOf;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.verifyZeroInteractions;
import static org.mockito.Mockito.when;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_HOME_DIRECTORY_PROPERTY;
import static org.mule.tck.ZipUtils.compress;

import org.mule.runtime.core.api.util.FileUtils;
import org.mule.runtime.deployment.model.api.policy.PolicyTemplateDescriptor;
import org.mule.runtime.module.deployment.impl.internal.policy.PolicyTemplateDescriptorFactory;
import org.mule.tck.ZipUtils;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.SystemPropertyTemporaryFolder;

import com.mulesoft.anypoint.tests.logger.DebugLine;
import com.mulesoft.anypoint.tests.logger.MockLogger;
import com.mulesoft.mule.runtime.gw.client.ApiPlatformClient;
import com.mulesoft.mule.runtime.gw.client.dto.PolicyTemplateDto;
import com.mulesoft.mule.runtime.gw.model.EmptyPolicySpecification;
import com.mulesoft.mule.runtime.gw.policies.template.exception.PolicyTemplateAssetException;
import com.mulesoft.mule.runtime.gw.policies.template.provider.FileSystemPolicyTemplateProvider;
import com.mulesoft.mule.runtime.gw.policies.template.provider.PolicyTemplateProvider;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.net.URISyntaxException;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class FileSystemPolicyTemplateProviderTestCase extends AbstractMuleTestCase {

  @Rule
  public TemporaryFolder muleHome = new SystemPropertyTemporaryFolder(MULE_HOME_DIRECTORY_PROPERTY);

  private PolicyTemplateDescriptor templateDescriptor;
  private ApiPlatformClient restClient;
  private PolicyTemplateDescriptorFactory templateDescriptorFactory;

  private PolicyTemplateProvider templateProvider;
  private MockLogger logger;

  private String VALID_YAML_RESOURCE = "/specs/client-id-enforcement.yaml";
  private String INVALID_YAML_RESOURCE = "/specs/invalid.yaml";

  @Before
  public void setUp() throws IllegalAccessException {
    templateDescriptorFactory = mock(PolicyTemplateDescriptorFactory.class);
    templateDescriptor = new PolicyTemplateDescriptor("name");
    templateDescriptor.setRootFolder(getPolicyTemplatesTempFolder());
    restClient = mock(ApiPlatformClient.class);
    logger = new MockLogger();

    when(templateDescriptorFactory.create(any(), any())).thenReturn(templateDescriptor);

    templateProvider = new FileSystemPolicyTemplateProvider(restClient);
    overrideVariable("templateDescriptorFactory").in(templateProvider).with(templateDescriptorFactory);
    overrideLogger().in(templateProvider).with(logger);
  }

  @Test
  public void provideHeavyWeightTemplate() throws IOException {
    createTemplateJarFile(String.format(POLICY_TEMPLATE_PATTERN, POLICY_TEMPLATE_KEY.getName()));
    createTemplateYamlFile(String.format(POLICY_YAML_PATTERN, POLICY_TEMPLATE_KEY.getName()), VALID_YAML_RESOURCE);

    PolicyTemplate policyTemplate = templateProvider.provide(POLICY_TEMPLATE_KEY);

    assertThat(policyTemplate.getKey(), is(POLICY_TEMPLATE_KEY));
    assertThat(policyTemplate.getTemplateFile(),
               is(new File(getPolicyTemplatesTempFolder(), policyTemplate.getKey().getName() + "/template.xml")));
    assertThat(policyTemplate.getTemplateDescriptor(), is(templateDescriptor));
    assertThat(getPolicyTemplatesTempFolder().listFiles(), arrayWithSize(1));
    verify(templateDescriptorFactory, times(1)).create(any(), any());
    verifyZeroInteractions(restClient);
  }

  @Test
  public void provideLightWeightTemplate() throws IOException {
    createTemplateJarFile(String.format(POLICY_TEMPLATE_LIGHT_PATTERN, POLICY_TEMPLATE_KEY.getName()));
    createTemplateYamlFile(String.format(POLICY_YAML_PATTERN, POLICY_TEMPLATE_KEY.getName()), VALID_YAML_RESOURCE);

    PolicyTemplate policyTemplate = templateProvider.provide(POLICY_TEMPLATE_KEY);

    assertThat(policyTemplate.getKey(), is(POLICY_TEMPLATE_KEY));
    assertThat(policyTemplate.getTemplateFile(),
               is(new File(getPolicyTemplatesTempFolder(), policyTemplate.getKey().getName() + "/template.xml")));
    assertThat(policyTemplate.getTemplateDescriptor(), is(templateDescriptor));
    assertThat(getPolicyTemplatesTempFolder().listFiles(), arrayWithSize(1));
    verify(templateDescriptorFactory, times(1)).create(any(), any());
    verifyZeroInteractions(restClient);
  }

  @Test
  public void provideDownloadedTemplate() throws IOException, URISyntaxException {
    File templateFile = createAnotherTemplateFile();
    File yamlFile = new File(getClass().getResource("/specs/client-id-enforcement.yaml").getFile());
    String downloadJarLink = "downloadJarLink";
    String downloadYamlLink = "downloadYamlLink";
    PolicyTemplateDto templateDTO = new PolicyTemplateDto();
    templateDTO.setJarDownloadLink(downloadJarLink);
    templateDTO.setYamlDownloadLink(downloadYamlLink);

    when(restClient.getPolicyTemplateMetadata(POLICY_TEMPLATE_KEY)).thenReturn(templateDTO);
    when(restClient.downloadTemplateAsset(downloadJarLink)).thenReturn(openInputStream(templateFile));
    when(restClient.downloadTemplateAsset(downloadYamlLink)).thenReturn(openInputStream(yamlFile));

    PolicyTemplate policyTemplate = templateProvider.provide(POLICY_TEMPLATE_KEY);

    assertThat(policyTemplate.getKey(), is(POLICY_TEMPLATE_KEY));
    assertThat(policyTemplate.getTemplateFile(),
               is(new File(getPolicyTemplatesTempFolder(), policyTemplate.getKey().getName() + "/template.xml")));
    assertThat(policyTemplate.getTemplateDescriptor(), is(templateDescriptor));
    assertThat(getPolicyTemplatesTempFolder().listFiles(), arrayWithSize(1));
    verify(restClient).downloadTemplateAsset(downloadJarLink);
    verify(restClient).downloadTemplateAsset(downloadYamlLink);
    verify(restClient).getPolicyTemplateMetadata(POLICY_TEMPLATE_KEY);
    verify(templateDescriptorFactory, times(1)).create(any(), any());
    verifyNoMoreInteractions(restClient);
  }

  @Test
  public void provideTwiceDoesNotRecreateTemplate() throws IOException {
    createTemplateJarFile(String.format(POLICY_TEMPLATE_PATTERN, POLICY_TEMPLATE_KEY.getName()));
    createTemplateYamlFile(String.format(POLICY_YAML_PATTERN, POLICY_TEMPLATE_KEY.getName()), VALID_YAML_RESOURCE);

    templateProvider.provide(POLICY_TEMPLATE_KEY);

    PolicyTemplate policyTemplate = templateProvider.provide(POLICY_TEMPLATE_KEY);

    assertThat(policyTemplate.getKey(), is(POLICY_TEMPLATE_KEY));
    assertThat(policyTemplate.getTemplateFile(),
               is(new File(getPolicyTemplatesTempFolder(), policyTemplate.getKey().getName() + "/template.xml")));
    assertThat(policyTemplate.getTemplateDescriptor(), is(templateDescriptor));
    assertThat(getPolicyTemplatesTempFolder().listFiles(), arrayWithSize(1));
    verify(templateDescriptorFactory, times(1)).create(any(), any());
    verifyZeroInteractions(restClient);
  }


  @Test
  public void provideRecreatesIfTempFolderDeleted() throws IOException {
    createTemplateJarFile(String.format(POLICY_TEMPLATE_PATTERN, POLICY_TEMPLATE_KEY.getName()));
    createTemplateYamlFile(String.format(POLICY_YAML_PATTERN, POLICY_TEMPLATE_KEY.getName()), VALID_YAML_RESOURCE);

    templateProvider.provide(POLICY_TEMPLATE_KEY);

    FileUtils.deleteTree(getPolicyTemplatesTempFolder());

    PolicyTemplate policyTemplate = templateProvider.provide(POLICY_TEMPLATE_KEY);

    assertThat(policyTemplate.getKey(), is(POLICY_TEMPLATE_KEY));
    assertThat(policyTemplate.getTemplateFile(),
               is(new File(getPolicyTemplatesTempFolder(), policyTemplate.getKey().getName() + "/template.xml")));
    assertThat(policyTemplate.getTemplateDescriptor(), is(templateDescriptor));
    assertThat(getPolicyTemplatesTempFolder().listFiles(), arrayWithSize(1));
    verify(templateDescriptorFactory, times(2)).create(any(), any());
    verifyZeroInteractions(restClient);
  }

  @Test
  public void templateMetadataError() throws IOException {
    when(restClient.getPolicyTemplateMetadata(POLICY_TEMPLATE_KEY)).thenThrow(new RuntimeException());

    try {
      templateProvider.provide(POLICY_TEMPLATE_KEY);
      fail("Template download should fail");
    } catch (PolicyTemplateAssetException ignored) {
    }

    assertThat(getPolicyTemplatesTempFolder().exists(), is(false));
    verify(restClient).getPolicyTemplateMetadata(POLICY_TEMPLATE_KEY);
    verifyNoMoreInteractions(restClient);
  }

  @Test
  public void templateJarDownloadError() throws IOException, URISyntaxException {
    String downloadLink = "jarDownloadLink";
    PolicyTemplateDto templateDto = new PolicyTemplateDto();
    templateDto.setJarDownloadLink(downloadLink);

    when(restClient.getPolicyTemplateMetadata(POLICY_TEMPLATE_KEY)).thenReturn(templateDto);
    when(restClient.downloadTemplateAsset(downloadLink)).thenThrow(new RuntimeException());

    try {
      templateProvider.provide(POLICY_TEMPLATE_KEY);
      fail("Template download should fail");
    } catch (PolicyTemplateAssetException ignored) {
    }

    assertThat(getPolicyTemplatesTempFolder().exists(), is(false));
    verify(restClient).getPolicyTemplateMetadata(POLICY_TEMPLATE_KEY);
    verify(restClient).downloadTemplateAsset(downloadLink);
    verifyNoMoreInteractions(restClient);
  }

  @Test
  public void templateYamlDownloadError() throws IOException, URISyntaxException {
    String downloadYamlLink = "yamlDownloadLink";
    String downloadJarLink = "jarDownloadLink";
    PolicyTemplateDto templateDto = new PolicyTemplateDto();
    templateDto.setYamlDownloadLink(downloadYamlLink);
    templateDto.setJarDownloadLink(downloadJarLink);

    when(restClient.getPolicyTemplateMetadata(POLICY_TEMPLATE_KEY)).thenReturn(templateDto);
    when(restClient.downloadTemplateAsset(downloadJarLink)).thenReturn(new FileInputStream(createAnotherTemplateFile()));
    when(restClient.downloadTemplateAsset(downloadYamlLink)).thenThrow(new RuntimeException());


    PolicyTemplate template = templateProvider.provide(POLICY_TEMPLATE_KEY);

    assertThat(template.getPolicySpecification(), is(instanceOf(EmptyPolicySpecification.class)));
    assertThat(getPolicyTemplatesTempFolder().exists(), is(true));
    verify(restClient).getPolicyTemplateMetadata(POLICY_TEMPLATE_KEY);
    verify(restClient).downloadTemplateAsset(downloadJarLink);
    verify(restClient).downloadTemplateAsset(downloadYamlLink);
    verifyNoMoreInteractions(restClient);
  }

  @Test
  public void invalidYamlParsing() throws IOException {
    createTemplateJarFile(String.format(POLICY_TEMPLATE_PATTERN, POLICY_TEMPLATE_KEY.getName()));
    createTemplateYamlFile(String.format(POLICY_YAML_PATTERN, POLICY_TEMPLATE_KEY.getName()), INVALID_YAML_RESOURCE);

    PolicyTemplate policyTemplate = templateProvider.provide(POLICY_TEMPLATE_KEY);

    assertThat(policyTemplate.getKey(), is(POLICY_TEMPLATE_KEY));
    assertThat(policyTemplate.getTemplateFile(),
               is(new File(getPolicyTemplatesTempFolder(), policyTemplate.getKey().getName() + "/template.xml")));
    assertThat(policyTemplate.getTemplateDescriptor(), is(templateDescriptor));
    assertThat(getPolicyTemplatesTempFolder().listFiles(), arrayWithSize(1));
    assertThat(policyTemplate.getPolicySpecification(), is(instanceOf(EmptyPolicySpecification.class)));
    verify(templateDescriptorFactory, times(1)).create(any(), any());
    verifyZeroInteractions(restClient);
  }

  @Test
  public void templateUnzipError() throws IOException {
    File templateJar = createInvalidTemplateFile(String.format(POLICY_TEMPLATE_PATTERN, POLICY_TEMPLATE_KEY.getName()));
    createTemplateYamlFile(String.format(POLICY_YAML_PATTERN, POLICY_TEMPLATE_KEY.getName()), VALID_YAML_RESOURCE);

    assertThat(templateJar.exists(), is(true));

    try {
      templateProvider.provide(POLICY_TEMPLATE_KEY);
      fail("Template unzipping should fail");
    } catch (PolicyTemplateAssetException exception) {
      assertThat(templateJar.exists(), is(false));
      assertThat(logger.lines().get(0),
                 is(new DebugLine("Policy template JAR {} was deleted successfully", templateJar.getName())));
    }
  }

  private void createTemplateJarFile(String templateFileName) throws IOException {
    File zipFile = new File(getPolicyTemplatesFolder(), templateFileName);
    File zipFileContent = new File(muleHome.getRoot(), "template.xml");

    zipFileContent.createNewFile();

    compress(zipFile, new ZipUtils.ZipResource[] {
        new ZipUtils.ZipResource(zipFileContent.getAbsolutePath(), zipFileContent.getName()),
    });
  }

  private File createTemplateYamlFile(String templateYamlFileName, String resource) throws IOException {
    File yamlFile = new File(getPolicyTemplatesFolder(), templateYamlFileName);
    yamlFile.createNewFile();
    copyInputStreamToFile(getClass().getResourceAsStream(resource), yamlFile);
    return yamlFile;
  }

  private File createAnotherTemplateFile() throws IOException {
    File zipFile = muleHome.newFile();
    File zipFileContent = new File(muleHome.getRoot(), "zipContent");

    zipFileContent.createNewFile();

    compress(zipFile, new ZipUtils.ZipResource[] {
        new ZipUtils.ZipResource(zipFileContent.getAbsolutePath(), zipFileContent.getName()),
    });

    return zipFile;
  }

  private File createInvalidTemplateFile(String templateFileName) throws IOException {
    File zipFile = new File(getPolicyTemplatesFolder(), templateFileName);
    File zipFileContent = new File(muleHome.getRoot(), "template.xml");

    zipFileContent.createNewFile();

    compress(zipFile, new ZipUtils.ZipResource[] {
        new ZipUtils.ZipResource(zipFileContent.getAbsolutePath(), zipFileContent.getName()),
    });

    RandomAccessFile raf = new RandomAccessFile(zipFile, "rw");
    raf.seek(20);
    raf.writeBytes("i_am_currupting_a_jar_file");
    raf.seek(189);
    raf.writeBytes("i_am_so_evil_hahahaha");
    raf.close();

    return zipFile;
  }
}
