/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.template.provider;

import static com.mulesoft.mule.runtime.gw.api.PolicyFolders.getPolicyTemplatesFolder;
import static org.apache.commons.io.FileUtils.copyInputStreamToFile;
import static org.mule.runtime.api.util.Preconditions.checkState;

import com.mulesoft.mule.runtime.gw.api.policy.PolicyTemplateKey;
import com.mulesoft.mule.runtime.gw.client.ApiPlatformClient;
import com.mulesoft.mule.runtime.gw.client.dto.PolicyTemplateDto;
import com.mulesoft.mule.runtime.gw.policies.template.exception.PolicyTemplateAssetException;

import java.io.File;
import java.io.InputStream;

/**
 * Looks for policy templates assets in policy-templates folder
 */
public class FileSystemPolicyTemplateProvider extends PolicyTemplateProvider {

  public static final String POLICY_TEMPLATE_PATTERN = "%s-mule-policy.jar";
  public static final String POLICY_TEMPLATE_LIGHT_PATTERN = "%s-mule-policy-light-package.jar";
  public static final String POLICY_YAML_PATTERN = "%s.yaml";

  private final File templatesFolder;
  private final ApiPlatformClient restClient;

  public FileSystemPolicyTemplateProvider(ApiPlatformClient restClient) {
    super();
    this.restClient = restClient;
    this.templatesFolder = getPolicyTemplatesFolder();
  }

  @Override
  protected PolicyTemplateAssets getPolicyTemplateAssets(PolicyTemplateKey templateKey) {
    PolicyTemplateAssets policyTemplateAssets = getTemplateFiles(templateKey);

    checkState(policyTemplateAssets.getTemplateJarFile().exists(), "Could not find policy Jar template " + templateKey.getName());

    return policyTemplateAssets;
  }

  private PolicyTemplateAssets getTemplateFiles(PolicyTemplateKey templateKey) {
    File templateFile = new File(templatesFolder, String.format(POLICY_TEMPLATE_PATTERN, templateKey.getName()));
    File lightTemplateFile = new File(templatesFolder, String.format(POLICY_TEMPLATE_LIGHT_PATTERN, templateKey.getName()));
    File yamlSpecFile = new File(templatesFolder, String.format(POLICY_YAML_PATTERN, templateKey.getName()));

    if ((!templateFile.exists() && !lightTemplateFile.exists())) {
      PolicyTemplateDto template;

      LOGGER.debug("Policy template {} assets are not present in the runtime. Fetching metadata for assets...", templateKey);

      try {
        template = restClient.getPolicyTemplateMetadata(templateKey);
      } catch (Exception e) {
        throw new PolicyTemplateAssetException("Unexpected error when requesting template metadata for policy "
            + templateKey.getName(), e);
      }

      templateFile = downloadFile(templateKey, template.getJarDownloadLink(), templateFile, true);
      yamlSpecFile = downloadFile(templateKey, template.getYamlDownloadLink(), yamlSpecFile, false);

      return new PolicyTemplateAssets(templateFile, yamlSpecFile);
    }

    return new PolicyTemplateAssets(templateFile.exists() ? templateFile : lightTemplateFile, yamlSpecFile);
  }

  private File downloadFile(PolicyTemplateKey templateKey, String link, File targetFile, boolean retry) {
    LOGGER.debug("Starting download of policy template asset {}", templateKey);

    try {
      InputStream inputStream = restClient.downloadTemplateAsset(link);

      copyInputStreamToFile(inputStream, targetFile);

      LOGGER.debug("Policy template {} successfully downloaded", templateKey);

    } catch (Exception e) {
      LOGGER.debug("Unable to save file {} from link {} ", targetFile.getName(), link);
      if (retry) {
        throw new PolicyTemplateAssetException("Unexpected error trying to download policy template asset"
            + templateKey.getName(), e);
      }
    }

    return targetFile;
  }
}
