/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.encryption;

import static com.google.common.collect.ImmutableMap.of;
import static com.mulesoft.mule.runtime.gw.internal.encryption.RuntimeEncrypterFactory.createDefaultRuntimeEncrypter;
import static java.lang.String.format;
import static java.util.Arrays.asList;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.contains;
import static org.junit.rules.ExpectedException.none;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.SystemProperty;

import com.mulesoft.mule.runtime.gw.internal.encryption.GatewayEncryptionException;
import com.mulesoft.mule.runtime.gw.internal.encryption.RuntimeEncrypter;
import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.model.PolicyProperty;
import com.mulesoft.mule.runtime.gw.model.PolicySpecification;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(Parameterized.class)
public class FullPolicyConfigurationEncrypterTestCase extends AbstractMuleTestCase {

  private static final String KEY = "Key";
  private static final String KEY_2 = "Key_2";
  private static final String KEY_LIST_STRING = "List_String";
  private static final String KEY_BOOLEAN = "Key_Boolean";
  private static final String KEY_KEYVALUE = "Keyvalue";
  private static final String KEY_KEYVALUE_2 = "Keyvalue_2";
  private static final String KEY_KEYVALUE_3 = "Keyvalue_3";
  private static final String KEY_LIST_KEYVALUES = "List_Keyvalues";
  private static final String KEY_NOT_DEFINED = "KeyNotDefinedInYaml";
  private static final String KEY_NULL_VALUE = "KeyWithNullValue";

  private static final String ENCRYPTION_KEY = "GatewayTeamKey00";

  private static final String ENCRYPTED_VALUE = "![z8p1poC0yyjUoTNzZhS2Xw==]";
  private static final String ENCRYPTED_VALUE_2 = "![dz3YF4DaHoJNwbdLD8Uzxg==]";
  private static final String ENCRYPTED_TEMPLATE_PLACEHOLDER = "${secure::%s}";
  private static final String SIMPLE_KEY = "mapKey";
  private static final String SIMPLE_KEY_2 = "mapKey2";
  private static final String SIMPLE_VALUE = "Test Value";
  private static final String SIMPLE_VALUE_2 = "Test Value2";
  private static final String ESCAPED_VALUE = "&quot;__&lt;Guns_&amp;&amp;_Roses&#39;&gt;__&quot;\"";
  private static final String ESCAPED_ENCRYTED_VALUE = "![1/qRVKESF3Y1oPkhwYLZ4fJmb4oJR5ZG4TUDLKkj9VM=]";
  private static final String HTML_ENCODED_EXPRESSION = "#[attributes.headers[&#39;client_secret&#39;]]";
  private static final String HTML_ENCODED_EXPRESSION_ENCRYPTED =
      "![bA9b4+qkgeX/zCyvDg58JSdwl/bKR9b6k2rgV+eY3ddpqh2SMAn3ZUc8fZQDyD3O]";


  private final PolicySpecification policySpecification;

  @Rule
  public ExpectedException expectedException = none();

  @Rule
  public SystemProperty encryptionKey = new SystemProperty("anypoint.platform.encryption_key", ENCRYPTION_KEY);

  private PolicyConfigurationEncrypter policyConfigurationEncrypter;

  @Parameterized.Parameters
  public static Collection<Object[]> testConfigurationData() {
    return asList(new Object[][] {
        {sensitiveSpecification()},
        {nonSensitiveSpecification()}
    });
  }

  public FullPolicyConfigurationEncrypterTestCase(PolicySpecification policySpecification) {
    this.policySpecification = policySpecification;
  }

  @Before
  public void setUp() {
    RuntimeEncrypter runtimeEncrypter = createDefaultRuntimeEncrypter();
    policyConfigurationEncrypter = new DefaultPolicyConfigurationEncrypter(runtimeEncrypter, false);
  }

  @Test
  public void encryptString() {
    PolicyConfigurationEncryptionResult result =
        policyConfigurationEncrypter.encryptConfiguration(policyDefinition(of(KEY, SIMPLE_VALUE,
                                                                              KEY_2, ESCAPED_VALUE)),
                                                          policySpecification);

    assertThat(result.getConfigurationProperties().isEncrypted(), is(true));
    assertThat(result.getConfigurationProperties().getConfiguration().get(KEY), is(ENCRYPTED_VALUE));
    assertThat(result.getConfigurationProperties().getConfiguration().get(KEY_2), is(ESCAPED_ENCRYTED_VALUE));
    assertThat(result.getConfigFileProperties().get(KEY), is(ENCRYPTED_VALUE));
    assertThat(result.getConfigFileProperties().get(KEY_2), is(ESCAPED_ENCRYTED_VALUE));
    assertThat(result.getTemplatePlaceholders().get(KEY), is(encryptedTemplatePlaceholder(KEY)));
    assertThat(result.getTemplatePlaceholders().get(KEY_2), is(encryptedTemplatePlaceholder(KEY_2)));
  }

  @Test
  public void encryptingExpressionWithHTMLEncodedCharacters() {
    PolicyConfigurationEncryptionResult result =
        policyConfigurationEncrypter.encryptConfiguration(policyDefinition(of(KEY, HTML_ENCODED_EXPRESSION)),
                                                          policySpecification);

    assertThat(result.getConfigurationProperties().isEncrypted(), is(true));

    assertThat(result.getConfigurationProperties().getConfiguration().get(KEY), is(HTML_ENCODED_EXPRESSION_ENCRYPTED));
    assertThat(result.getTemplatePlaceholders().get(KEY), is(encryptedTemplatePlaceholder(KEY)));
    assertThat(result.getConfigFileProperties().get(KEY), is(HTML_ENCODED_EXPRESSION_ENCRYPTED));
  }

  @Test
  public void encryptBoolean() {
    PolicyConfigurationEncryptionResult result =
        policyConfigurationEncrypter.encryptConfiguration(policyDefinition(of(KEY_BOOLEAN, true)), policySpecification);

    assertThat(result.getConfigurationProperties().isEncrypted(), is(false));
    assertThat(result.getConfigurationProperties().getConfiguration().get(KEY_BOOLEAN), is(true));
    assertThat(result.getConfigFileProperties().get(KEY_BOOLEAN), nullValue());
    assertThat(result.getTemplatePlaceholders().get(KEY_BOOLEAN), is(true));
  }

  @Test
  public void encryptNotDefinedKey() {
    PolicyConfigurationEncryptionResult result =
        policyConfigurationEncrypter.encryptConfiguration(policyDefinition(of(KEY_NOT_DEFINED, SIMPLE_VALUE)),
                                                          policySpecification);

    assertThat(result.getConfigurationProperties().isEncrypted(), is(false));
    assertThat(result.getConfigurationProperties().getConfiguration().get(KEY_NOT_DEFINED), is(SIMPLE_VALUE));
    assertThat(result.getConfigFileProperties().get(KEY_NOT_DEFINED), nullValue());
    assertThat(result.getTemplatePlaceholders().get(KEY_NOT_DEFINED), is(SIMPLE_VALUE));
  }

  @Test
  public void encryptListOfStrings() {
    PolicyConfigurationEncryptionResult result =
        policyConfigurationEncrypter
            .encryptConfiguration(policyDefinition(of(KEY_LIST_STRING, asList(SIMPLE_VALUE, SIMPLE_VALUE_2, ESCAPED_VALUE))),
                                  policySpecification);

    assertThat(result.getConfigurationProperties().isEncrypted(), is(true));
    assertThat((List<String>) result.getConfigurationProperties().getConfiguration().get(KEY_LIST_STRING),
               contains(ENCRYPTED_VALUE, ENCRYPTED_VALUE_2, ESCAPED_ENCRYTED_VALUE));
    assertThat((List<String>) result.getTemplatePlaceholders().get(KEY_LIST_STRING),
               contains(encryptedTemplatePlaceholder(KEY_LIST_STRING + ".0"),
                        encryptedTemplatePlaceholder(KEY_LIST_STRING + ".1"),
                        encryptedTemplatePlaceholder(KEY_LIST_STRING + ".2")));
    assertThat(((Map) result.getConfigFileProperties().get(KEY_LIST_STRING)).get("0"), is(ENCRYPTED_VALUE));
    assertThat(((Map) result.getConfigFileProperties().get(KEY_LIST_STRING)).get("1"), is(ENCRYPTED_VALUE_2));
    assertThat(((Map) result.getConfigFileProperties().get(KEY_LIST_STRING)).get("2"), is(ESCAPED_ENCRYTED_VALUE));
  }

  @Test
  public void encryptKeyvalue() {
    Map<String, String> propertyMap = of("key", SIMPLE_KEY, "value", SIMPLE_VALUE);
    Map<String, String> propertyMap2 = of("key", SIMPLE_KEY, "value", ESCAPED_VALUE);
    Map<String, String> propertyMap3 = new HashMap<>();
    propertyMap3.put("key", SIMPLE_KEY);
    propertyMap3.put("value", null);

    PolicyConfigurationEncryptionResult result =
        policyConfigurationEncrypter.encryptConfiguration(policyDefinition(of(KEY_KEYVALUE, propertyMap,
                                                                              KEY_KEYVALUE_2, propertyMap2,
                                                                              KEY_KEYVALUE_3, propertyMap3)),
                                                          policySpecification);

    assertThat(result.getConfigurationProperties().isEncrypted(), is(true));
    assertKeyValue((Map<String, Object>) result.getConfigurationProperties().getConfiguration().get(KEY_KEYVALUE), SIMPLE_KEY,
                   ENCRYPTED_VALUE);
    assertKeyValue((Map<String, Object>) result.getConfigurationProperties().getConfiguration().get(KEY_KEYVALUE_2), SIMPLE_KEY,
                   ESCAPED_ENCRYTED_VALUE);
    assertKeyValue((Map) result.getTemplatePlaceholders().get(KEY_KEYVALUE), SIMPLE_KEY,
                   encryptedTemplatePlaceholder(KEY_KEYVALUE, SIMPLE_KEY));
    assertKeyValue((Map) result.getTemplatePlaceholders().get(KEY_KEYVALUE_2), SIMPLE_KEY,
                   encryptedTemplatePlaceholder(KEY_KEYVALUE_2, SIMPLE_KEY));
    assertKeyValue((Map<String, Object>) result.getConfigFileProperties().get(KEY_KEYVALUE), SIMPLE_KEY,
                   ENCRYPTED_VALUE);
    assertKeyValue((Map<String, Object>) result.getConfigFileProperties().get(KEY_KEYVALUE_2), SIMPLE_KEY,
                   ESCAPED_ENCRYTED_VALUE);
  }

  @Test
  public void encryptListOfKeyvalues() {
    Map<String, String> propertyMap = of("key", SIMPLE_KEY, "value", SIMPLE_VALUE);
    Map<String, String> propertyMap2 = of("key", SIMPLE_KEY_2, "value", ESCAPED_VALUE);

    PolicyConfigurationEncryptionResult result =
        policyConfigurationEncrypter
            .encryptConfiguration(policyDefinition(of(KEY_LIST_KEYVALUES, asList(propertyMap, propertyMap2))),
                                  policySpecification);

    assertThat(result.getConfigurationProperties().isEncrypted(), is(true));
    assertKeyValue((Map<String, Object>) ((List) result.getConfigurationProperties().getConfiguration().get(KEY_LIST_KEYVALUES))
        .get(0), SIMPLE_KEY_2, ESCAPED_ENCRYTED_VALUE);
    assertKeyValue((Map<String, Object>) ((List) result.getConfigurationProperties().getConfiguration().get(KEY_LIST_KEYVALUES))
        .get(1), SIMPLE_KEY, ENCRYPTED_VALUE);
    assertKeyValue((Map<String, Object>) ((List) result.getTemplatePlaceholders().get(KEY_LIST_KEYVALUES))
        .get(0), SIMPLE_KEY_2, encryptedTemplatePlaceholder(KEY_LIST_KEYVALUES, SIMPLE_KEY_2));
    assertKeyValue((Map<String, Object>) ((List) result.getTemplatePlaceholders().get(KEY_LIST_KEYVALUES))
        .get(1), SIMPLE_KEY, encryptedTemplatePlaceholder(KEY_LIST_KEYVALUES, SIMPLE_KEY));
    assertThat(((Map<String, Object>) result.getConfigFileProperties().get(KEY_LIST_KEYVALUES)).get(SIMPLE_KEY),
               is(ENCRYPTED_VALUE));
    assertThat(((Map<String, Object>) result.getConfigFileProperties().get(KEY_LIST_KEYVALUES)).get(SIMPLE_KEY_2),
               is(ESCAPED_ENCRYTED_VALUE));
  }

  @Test
  public void encryptNullValue() {
    Map<String, Object> nullConfigValue = new HashMap<>();
    nullConfigValue.put(KEY_NULL_VALUE, null);

    PolicyConfigurationEncryptionResult result =
        policyConfigurationEncrypter.encryptConfiguration(policyDefinition(nullConfigValue), policySpecification);

    assertThat(result.getConfigurationProperties().getConfiguration().containsKey(KEY_NULL_VALUE), is(true));
    assertThat(result.getConfigurationProperties().getConfiguration().get(KEY_NULL_VALUE), nullValue());
    assertThat(result.getTemplatePlaceholders().containsKey(KEY_NULL_VALUE), is(true));
    assertThat(result.getTemplatePlaceholders().get(KEY_NULL_VALUE), nullValue());
    assertThat(result.getConfigFileProperties().isEmpty(), is(true));
  }

  @Test
  public void encrypterFailsToEncrypt() {
    RuntimeEncrypter invalidEncrypter = mock(RuntimeEncrypter.class);
    when(invalidEncrypter.encrypt(any())).thenThrow(new GatewayEncryptionException(""));
    policyConfigurationEncrypter = new DefaultPolicyConfigurationEncrypter(invalidEncrypter, false);

    expectedException.expect(GatewayEncryptionException.class);

    policyConfigurationEncrypter.encryptConfiguration(policyDefinition(of(KEY, SIMPLE_VALUE)), policySpecification);
  }

  private PolicyDefinition policyDefinition(Map<String, Object> configData) {
    PolicyDefinition policyDefinition = mock(PolicyDefinition.class);

    when(policyDefinition.getConfigurationData()).thenReturn(new PolicyConfiguration(configData));

    return policyDefinition;
  }

  private static PolicySpecification sensitiveSpecification() {
    return mockSpecification(true);
  }

  private static PolicySpecification nonSensitiveSpecification() {
    return mockSpecification(false);
  }

  private static PolicySpecification mockSpecification(boolean sensitive) {
    PolicySpecification policySpecification = mock(PolicySpecification.class);

    when(policySpecification.getConfiguration()).thenReturn(buildMockedPolicyProperties(sensitive));
    when(policySpecification.isValid()).thenReturn(true);
    return policySpecification;
  }

  private static List<PolicyProperty> buildMockedPolicyProperties(boolean sensitive) {
    PolicyProperty policyProperty1 = new PolicyProperty();
    policyProperty1.setPropertyName(KEY);
    policyProperty1.setSensitive(sensitive);
    policyProperty1.setType("string");
    policyProperty1.setAllowMultiple(false);

    PolicyProperty policyProperty2 = new PolicyProperty();
    policyProperty2.setPropertyName(KEY_2);
    policyProperty2.setSensitive(sensitive);
    policyProperty2.setType("string");
    policyProperty2.setAllowMultiple(false);

    PolicyProperty policyProperty3 = new PolicyProperty();
    policyProperty3.setPropertyName(KEY_BOOLEAN);
    policyProperty3.setSensitive(sensitive);
    policyProperty3.setType("boolean");
    policyProperty3.setAllowMultiple(false);

    PolicyProperty policyProperty4 = new PolicyProperty();
    policyProperty4.setPropertyName(KEY_KEYVALUE);
    policyProperty4.setSensitive(sensitive);
    policyProperty4.setType("keyvalues");
    policyProperty4.setAllowMultiple(false);

    PolicyProperty policyProperty5 = new PolicyProperty();
    policyProperty5.setPropertyName(KEY_LIST_KEYVALUES);
    policyProperty5.setSensitive(sensitive);
    policyProperty5.setType("keyvalues");
    policyProperty5.setAllowMultiple(true);

    PolicyProperty policyProperty6 = new PolicyProperty();
    policyProperty6.setPropertyName(KEY_LIST_STRING);
    policyProperty6.setSensitive(sensitive);
    policyProperty6.setType("string");
    policyProperty6.setAllowMultiple(true);

    PolicyProperty policyProperty7 = new PolicyProperty();
    policyProperty7.setPropertyName(KEY_KEYVALUE_2);
    policyProperty7.setSensitive(sensitive);
    policyProperty7.setType("keyvalues");
    policyProperty7.setAllowMultiple(false);

    return asList(policyProperty1, policyProperty2, policyProperty3, policyProperty4, policyProperty5, policyProperty6,
                  policyProperty7);
  }

  private void assertKeyValue(Map<String, Object> keyvalue, String key, String value) {
    assertThat(keyvalue.get("key"), is(key));
    assertThat(keyvalue.get("value"), is(value));
  }

  private String encryptedTemplatePlaceholder(String property) {
    return format(ENCRYPTED_TEMPLATE_PLACEHOLDER, property);
  }

  private String encryptedTemplatePlaceholder(String property, String innerProperty) {
    return format(ENCRYPTED_TEMPLATE_PLACEHOLDER, property + "." + innerProperty);
  }
}
