/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.factory;

import static org.mule.runtime.api.util.MultiMap.emptyMultiMap;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_HOME_DIRECTORY_PROPERTY;

import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.RESOURCE_POINTCUTS;
import static com.mulesoft.mule.runtime.gw.policies.pointcut.PointcutTestUtils.listenerPointcutParameters;
import static com.mulesoft.mule.runtime.gw.policies.pointcut.PointcutTestUtils.nonHttpPointcutParameters;
import static com.mulesoft.mule.runtime.gw.reflection.VariableOverride.overrideLogger;

import static java.util.Collections.emptyMap;

import static com.google.common.collect.ImmutableMap.of;
import static javax.ws.rs.HttpMethod.GET;
import static javax.ws.rs.HttpMethod.HEAD;
import static jersey.repackaged.com.google.common.collect.Lists.newArrayList;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasItem;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsInstanceOf.instanceOf;
import static org.junit.Assert.assertThat;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.notification.NotificationListener;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.core.api.construct.Flow;
import org.mule.runtime.core.api.policy.PolicyParametrization;
import org.mule.runtime.deployment.model.api.application.Application;
import org.mule.runtime.http.api.domain.CaseInsensitiveMultiMap;
import org.mule.runtime.policy.api.PolicyPointcut;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.SystemPropertyTemporaryFolder;

import com.mulesoft.anypoint.tests.logger.DebugLine;
import com.mulesoft.anypoint.tests.logger.MockLogger;
import com.mulesoft.mule.runtime.gw.api.policy.HttpResourcePointcut;
import com.mulesoft.mule.runtime.gw.model.ApiImplementation;
import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.OfflinePolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.pointcut.CompositePointcut;
import com.mulesoft.mule.runtime.gw.policies.pointcut.HttpHeaderPointcut;

import java.io.File;
import java.util.Map;

import javax.xml.namespace.QName;

import com.google.common.collect.ImmutableMap;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import io.qameta.allure.Issue;

public class PolicyParametrizationFactoryTestCase extends AbstractMuleTestCase {

  private static final String FLOW_NAME = "flowName";
  private static final String INVALID_FLOW_NAME = "invalidFlowName";
  private static final String IS_WSDL_ENDPOINT = "isWsdlEndpoint";
  private static final String API_ID = "apiId";
  private static final String HDP_SERVICE_HEADER = "X-MULE-HDP-SERVICE";
  private static final String HDP_SERVICE = "svc";
  private static final String PATH = "/base/path/resource/1";
  private static final String MASKED_PATH = "/resource/1";

  @Rule
  public TemporaryFolder muleHome = new SystemPropertyTemporaryFolder(MULE_HOME_DIRECTORY_PROPERTY);

  private File file;
  private NotificationListener listener;
  private ApiImplementation apiImplementation;
  private PolicyDefinition policyDefinition;
  private MockLogger logger;

  private final PolicyParametrizationFactory factory = new PolicyParametrizationFactory();

  @Before
  public void setUp() {
    file = mock(File.class);
    listener = mock(NotificationListener.class);
    logger = new MockLogger();
    overrideLogger().in(factory).with(logger);
    apiImplementation = getApiImplementation();
  }

  @Test
  public void createWithOnlyApiPointcut() {
    policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, null, 1, new PolicyConfiguration(emptyMap()));

    PolicyParametrization parametrization =
        factory.create(policyDefinition, apiImplementation, file, null, false, newArrayList(listener));

    assertThat(parametrization.getPolicyPointcut(), instanceOf(CompositePointcut.class));
    assertBasicPointcutPresent(parametrization.getPolicyPointcut());
    assertResourcePointcutsNotPresent(parametrization.getPolicyPointcut());
    assertThat(parametrization.getId(), is(policyDefinition.getName() + " @ app-" + FLOW_NAME));
    assertThat(parametrization.getOrder(), is(policyDefinition.getOrder()));
    assertThat(parametrization.getConfig(), is(file));
    assertThat(parametrization.getParameters().get(API_ID), is(API_KEY.id().toString()));
    assertThat(parametrization.getParameters().get(IS_WSDL_ENDPOINT), is("false"));
    assertThat(parametrization.getNotificationListeners(), hasSize(1));
    assertThat(parametrization.getNotificationListeners(), hasItem(listener));
    assertThat(logger.lines(), hasSize(1));
    assertThat(logger.lines().get(0),
               is(new DebugLine("There are no resource pointcuts defined for policy {}. It will be applied to flow {}",
                                policyDefinition.getName(), FLOW_NAME)));
  }

  @Test
  public void createWithApiAndResourcePointcuts() {
    policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, RESOURCE_POINTCUTS, 1, new PolicyConfiguration(emptyMap()));

    PolicyParametrization parametrization =
        factory.create(policyDefinition, apiImplementation, file, null, false, newArrayList(listener));

    assertThat(parametrization.getPolicyPointcut(), instanceOf(CompositePointcut.class));
    assertBasicPointcutPresent(parametrization.getPolicyPointcut());
    assertResourcePointcutsPresent(parametrization.getPolicyPointcut());
    assertThat(parametrization.getId(), is(policyDefinition.getName() + " @ app-" + FLOW_NAME));
    assertThat(parametrization.getOrder(), is(policyDefinition.getOrder()));
    assertThat(parametrization.getConfig(), is(file));
    assertThat(parametrization.getParameters().get(API_ID), is(API_KEY.id().toString()));
    assertThat(parametrization.getNotificationListeners(), hasSize(1));
    assertThat(parametrization.getNotificationListeners(), hasItem(listener));
  }

  @Test
  public void createWithIsWsdlEndpointAsString() {
    Map<String, Object> configData = ImmutableMap.of(IS_WSDL_ENDPOINT, "true");
    policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, null, 1, new PolicyConfiguration(configData));

    PolicyParametrization parametrization =
        factory.create(policyDefinition, apiImplementation, file, null, false, newArrayList(listener));

    assertThat(parametrization.getPolicyPointcut(), instanceOf(CompositePointcut.class));
    assertThat(parametrization.getId(), is(policyDefinition.getName() + " @ app-" + FLOW_NAME));
    assertThat(parametrization.getOrder(), is(policyDefinition.getOrder()));
    assertThat(parametrization.getConfig(), is(file));
    assertThat(parametrization.getParameters().get(API_ID), is(API_KEY.id().toString()));
    assertThat(parametrization.getParameters().get(IS_WSDL_ENDPOINT), is("true"));
    assertThat(parametrization.getNotificationListeners(), hasSize(1));
    assertThat(parametrization.getNotificationListeners(), hasItem(listener));
    assertThat(logger.lines(), hasSize(1));
    assertThat(logger.lines().get(0),
               is(new DebugLine("There are no resource pointcuts defined for policy {}. It will be applied to flow {}",
                                policyDefinition.getName(), FLOW_NAME)));
  }

  @Test
  public void createWithIsWsdlEndpointAsBoolean() {
    Map<String, Object> configData = ImmutableMap.of(IS_WSDL_ENDPOINT, true);
    policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, null, 1, new PolicyConfiguration(configData));

    PolicyParametrization parametrization =
        factory.create(policyDefinition, apiImplementation, file, null, false, newArrayList(listener));

    assertThat(parametrization.getPolicyPointcut(), instanceOf(CompositePointcut.class));
    assertThat(parametrization.getId(), is(policyDefinition.getName() + " @ app-" + FLOW_NAME));
    assertThat(parametrization.getOrder(), is(policyDefinition.getOrder()));
    assertThat(parametrization.getConfig(), is(file));
    assertThat(parametrization.getParameters().get(API_ID), is(API_KEY.id().toString()));
    assertThat(parametrization.getParameters().get(IS_WSDL_ENDPOINT), is("true"));
    assertThat(parametrization.getNotificationListeners(), hasSize(1));
    assertThat(parametrization.getNotificationListeners(), hasItem(listener));
    assertThat(logger.lines(), hasSize(1));
    assertThat(logger.lines().get(0),
               is(new DebugLine("There are no resource pointcuts defined for policy {}. It will be applied to flow {}",
                                policyDefinition.getName(), FLOW_NAME)));
  }

  @Test
  public void createWithHdpPointcut() {
    apiImplementation = getApiImplementation("svc");
    policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, null, 1, new PolicyConfiguration(emptyMap()));

    PolicyParametrization parametrization =
        factory.create(policyDefinition, apiImplementation, file, null, false, newArrayList(listener));

    assertThat(parametrization.getPolicyPointcut(), instanceOf(CompositePointcut.class));
    assertHdpPointcutPresent(parametrization.getPolicyPointcut());
    assertHdpResourcePointcutsNotPresent(parametrization.getPolicyPointcut());
    assertThat(parametrization.getId(), is(policyDefinition.getName() + " @ app-" + FLOW_NAME));
    assertThat(parametrization.getOrder(), is(policyDefinition.getOrder()));
    assertThat(parametrization.getConfig(), is(file));
    assertThat(parametrization.getParameters().get(API_ID), is(API_KEY.id().toString()));
    assertThat(parametrization.getParameters().get(IS_WSDL_ENDPOINT), is("false"));
    assertThat(parametrization.getNotificationListeners(), hasSize(1));
    assertThat(parametrization.getNotificationListeners(), hasItem(listener));
    assertThat(logger.lines(), hasSize(2));
    assertThat(logger.lines().get(0),
               is(new DebugLine("Injecting header pointcut {} for service {}",
                                new HttpHeaderPointcut(POLICY_ID, HDP_SERVICE_HEADER, HDP_SERVICE).toString(), HDP_SERVICE)));
    assertThat(logger.lines().get(1),
               is(new DebugLine("There are no resource pointcuts defined for policy {}. It will be applied to flow {}",
                                policyDefinition.getName(), FLOW_NAME)));
  }

  @Test
  public void createWithHdpAndResourcePointcuts() {
    apiImplementation = getApiImplementation("svc");
    policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, RESOURCE_POINTCUTS, 1, new PolicyConfiguration(emptyMap()));

    PolicyParametrization parametrization =
        factory.create(policyDefinition, apiImplementation, file, null, false, newArrayList(listener));

    assertThat(parametrization.getPolicyPointcut(), instanceOf(CompositePointcut.class));
    assertHdpPointcutPresent(parametrization.getPolicyPointcut());
    assertHdpResourcePointcutsPresent(parametrization.getPolicyPointcut());
    assertThat(parametrization.getId(), is(policyDefinition.getName() + " @ app-" + FLOW_NAME));
    assertThat(parametrization.getOrder(), is(policyDefinition.getOrder()));
    assertThat(parametrization.getConfig(), is(file));
    assertThat(parametrization.getParameters().get(API_ID), is(API_KEY.id().toString()));
    assertThat(parametrization.getNotificationListeners(), hasSize(1));
    assertThat(parametrization.getNotificationListeners(), hasItem(listener));
    assertThat(logger.lines(), hasSize(1));
    assertThat(logger.lines().get(0),
               is(new DebugLine("Injecting header pointcut {} for service {}",
                                new HttpHeaderPointcut(POLICY_ID, HDP_SERVICE_HEADER, HDP_SERVICE).toString(), HDP_SERVICE)));
  }

  @Test
  public void createWithResourcePointcutUsingMaskedPath() {
    apiImplementation = getApiImplementation(null, true);
    policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY,
                             newArrayList(new HttpResourcePointcut("/resource/.*", ".*")), 1,
                             new PolicyConfiguration(emptyMap()));

    PolicyParametrization parametrization =
        factory.create(policyDefinition, apiImplementation, file, null, false, newArrayList(listener));

    assertThat(parametrization.getPolicyPointcut(), instanceOf(CompositePointcut.class));
    assertMaskedPathResourcePointcutsPresent(parametrization.getPolicyPointcut());
    assertThat(parametrization.getId(), is(policyDefinition.getName() + " @ app-" + FLOW_NAME));
    assertThat(parametrization.getOrder(), is(policyDefinition.getOrder()));
    assertThat(parametrization.getConfig(), is(file));
    assertThat(parametrization.getParameters().get(API_ID), is(API_KEY.id().toString()));
    assertThat(parametrization.getNotificationListeners(), hasSize(1));
    assertThat(parametrization.getNotificationListeners(), hasItem(listener));
  }

  @Test
  @Issue("W-12010579")
  public void sameOfflinePolicyAppliedToEquallyNamedFlowsOnDifferentApps() {
    apiImplementation = getApiImplementation(null, "app1", true);
    policyDefinition = new OfflinePolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY,
                                                   newArrayList(new HttpResourcePointcut("/resource/.*", ".*")), 1,
                                                   new PolicyConfiguration(emptyMap()));

    PolicyParametrization parametrization =
        factory.create(policyDefinition, apiImplementation, file, null, false, newArrayList(listener));

    ApiImplementation apiImplementation2 = getApiImplementation(null, "apps", true);
    PolicyDefinition policyDefinition2 = new OfflinePolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY,
                                                                     newArrayList(new HttpResourcePointcut("/resource/.*", ".*")),
                                                                     1,
                                                                     new PolicyConfiguration(emptyMap()));

    PolicyParametrization parametrization2 =
        factory.create(policyDefinition2, apiImplementation2, file, null, false, newArrayList(listener));

    assertThat(parametrization.getId(), not(equalTo(parametrization2.getId())));
  }

  private void assertBasicPointcutPresent(PolicyPointcut policyPointcut) {
    Map<QName, Object> annotations = ImmutableMap.of();
    assertThat(policyPointcut.matches(listenerPointcutParameters(FLOW_NAME, PATH, GET, annotations)), is(true));
    assertThat(policyPointcut.matches(listenerPointcutParameters(INVALID_FLOW_NAME, PATH, GET, annotations)), is(false));
    assertThat(policyPointcut.matches(nonHttpPointcutParameters(FLOW_NAME)), is(false));
  }

  private void assertResourcePointcutsPresent(PolicyPointcut policyPointcut) {
    Map<QName, Object> annotations = ImmutableMap.of();
    assertThat(policyPointcut.matches(listenerPointcutParameters(FLOW_NAME, PATH, HEAD, annotations)), is(false));
    assertThat(policyPointcut.matches(listenerPointcutParameters(FLOW_NAME, PATH, GET, annotations)), is(true));
  }

  private void assertMaskedPathResourcePointcutsPresent(PolicyPointcut policyPointcut) {
    Map<QName, Object> annotations = ImmutableMap.of();
    assertThat(policyPointcut
        .matches(listenerPointcutParameters(FLOW_NAME, PATH, HEAD, MASKED_PATH, emptyMultiMap(), annotations)),
               is(true));
    assertThat(policyPointcut
        .matches(listenerPointcutParameters(FLOW_NAME, "/resource/1", HEAD, "/1", emptyMultiMap(), annotations)),
               is(false));
    assertThat(policyPointcut
        .matches(listenerPointcutParameters(INVALID_FLOW_NAME, PATH, HEAD, MASKED_PATH, emptyMultiMap(), annotations)),
               is(false));
  }

  private void assertResourcePointcutsNotPresent(PolicyPointcut policyPointcut) {
    Map<QName, Object> annotations = ImmutableMap.of();
    assertThat(policyPointcut.matches(listenerPointcutParameters(FLOW_NAME, PATH, HEAD, annotations)), is(true));
    assertThat(policyPointcut.matches(listenerPointcutParameters(FLOW_NAME, PATH, GET, annotations)), is(true));
  }

  private void assertHdpPointcutPresent(PolicyPointcut policyPointcut) {
    Map<QName, Object> annotations = ImmutableMap.of();
    assertThat(policyPointcut.matches(listenerPointcutParameters(FLOW_NAME, PATH, GET, null, getHdpHeader(), annotations)),
               is(true));
    assertThat(policyPointcut
        .matches(listenerPointcutParameters(INVALID_FLOW_NAME, PATH, GET, null, getHdpHeader(), annotations)),
               is(false));
    assertThat(policyPointcut.matches(nonHttpPointcutParameters(FLOW_NAME)), is(false));
  }

  private void assertHdpResourcePointcutsPresent(PolicyPointcut policyPointcut) {
    Map<QName, Object> annotations = ImmutableMap.of();
    assertThat(policyPointcut
        .matches(listenerPointcutParameters(FLOW_NAME, PATH, HEAD, null, getHdpHeader(), annotations)),
               is(false));
    assertThat(policyPointcut.matches(listenerPointcutParameters(FLOW_NAME, PATH, GET, null, getHdpHeader(), annotations)),
               is(true));
  }

  private void assertHdpResourcePointcutsNotPresent(PolicyPointcut policyPointcut) {
    Map<QName, Object> annotations = ImmutableMap.of();
    assertThat(policyPointcut
        .matches(listenerPointcutParameters(FLOW_NAME, PATH, HEAD, null, getHdpHeader(), annotations)),
               is(true));
    assertThat(policyPointcut.matches(listenerPointcutParameters(FLOW_NAME, PATH, GET, null, getHdpHeader(), annotations)),
               is(true));
  }

  private MultiMap<String, String> getHdpHeader() {
    return new CaseInsensitiveMultiMap(new MultiMap<>(of(HDP_SERVICE_HEADER, HDP_SERVICE)));
  }

  private ApiImplementation getApiImplementation() {
    return getApiImplementation(null);
  }

  private ApiImplementation getApiImplementation(String hdpService) {
    return getApiImplementation(hdpService, false);
  }

  private ApiImplementation getApiImplementation(String hdpService, boolean ignoreBasePathOnResourceLevel) {
    return getApiImplementation(hdpService, "app", ignoreBasePathOnResourceLevel);
  }

  private ApiImplementation getApiImplementation(String hdpService, String appName, boolean ignoreBasePathOnResourceLevel) {
    Application application = mock(Application.class);
    when(application.getArtifactName()).thenReturn(appName);
    Flow flow = mock(Flow.class);
    when(flow.getName()).thenReturn(FLOW_NAME);
    return new ApiImplementation(API_KEY, application, flow, hdpService, ignoreBasePathOnResourceLevel);
  }
}
