/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.service.detection;

import static com.google.common.collect.Lists.newArrayList;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY_2;
import static com.mulesoft.mule.runtime.gw.policies.service.DeploymentStatusTestFactory.applied;
import static com.mulesoft.mule.runtime.gw.policies.service.DeploymentStatusTestFactory.deploymentFailed;
import static com.mulesoft.mule.runtime.gw.policies.service.DeploymentStatusTestFactory.downloadFailed;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static java.util.Collections.singletonMap;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsInstanceOf.instanceOf;

import com.mulesoft.mule.runtime.gw.api.policy.HttpResourcePointcut;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.service.detection.change.ChangeType;
import com.mulesoft.mule.runtime.gw.policies.service.detection.change.PolicyAdded;
import com.mulesoft.mule.runtime.gw.policies.service.detection.change.PolicyRevert;
import com.mulesoft.mule.runtime.gw.policies.service.detection.change.PolicyUpdate;
import com.mulesoft.mule.runtime.gw.policies.service.detection.change.PolicyReorder;
import com.mulesoft.mule.runtime.gw.policies.service.detection.change.PolicyUnmodified;
import com.mulesoft.mule.runtime.gw.policies.service.detection.change.RecoverableChangeType;

import org.junit.Test;

public class PolicyChangeInspectorTestCase {

  private static final PolicyDefinition definition =
      new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, newArrayList(), 1, emptyMap());
  private static final PolicyDefinition differentDefinition =
      new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, newArrayList(), 1, singletonMap("param", "value"));

  PolicyChangeInspector inspector = new PolicyChangeInspector();

  @Test
  public void noChangesWhenSuccessfullyApplied() {
    ChangeType change = inspector.detectChange(applied(definition), definition);
    assertThat(change, instanceOf(PolicyUnmodified.class));
    assertThat(change.getDefinition(), is(definition));
  }

  @Test
  public void noChangesWhenDeploymentFailed() {
    ChangeType change = inspector.detectChange(deploymentFailed(definition), definition);
    assertThat(change, instanceOf(PolicyUnmodified.class));
    assertThat(change.getDefinition(), is(definition));
  }

  @Test
  public void noChangesWhenDeploymentFailedWithPreviousSuccess() {
    ChangeType change = inspector.detectChange(deploymentFailed(definition, differentDefinition), differentDefinition);
    assertThat(change, instanceOf(PolicyUnmodified.class));
    assertThat(change.getDefinition(), is(differentDefinition));
  }

  @Test
  public void reorder() {
    PolicyDefinition reorderedDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, newArrayList(), 2, emptyMap());
    ChangeType change = inspector.detectChange(applied(definition), reorderedDefinition);
    assertThat(change, instanceOf(PolicyReorder.class));
    assertThat(change.getDefinition(), is(reorderedDefinition));
    assertThat(((RecoverableChangeType) change).getAppliedDefinition(), is(definition));
  }

  @Test
  public void reorderOfApplied() {
    PolicyDefinition reorderedDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, newArrayList(), 2, emptyMap());
    ChangeType change = inspector.detectChange(deploymentFailed(definition, differentDefinition), reorderedDefinition);
    assertThat(change, instanceOf(PolicyReorder.class));
    assertThat(change.getDefinition(), is(reorderedDefinition));
    assertThat(((RecoverableChangeType) change).getAppliedDefinition(), is(definition));
  }

  @Test
  public void updateConfig() {
    ChangeType change = inspector.detectChange(applied(definition), differentDefinition);
    assertThat(change, instanceOf(PolicyUpdate.class));
    assertThat(change.getDefinition(), is(differentDefinition));
    assertThat(((RecoverableChangeType) change).getAppliedDefinition(), is(definition));
  }

  @Test
  public void updateOrderPlusAnotherChange() {
    PolicyDefinition newDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, newArrayList(), 2, singletonMap("param", "value"));
    ChangeType change = inspector.detectChange(applied(definition), newDefinition);
    assertThat(change, instanceOf(PolicyUpdate.class));
    assertThat(change.getDefinition(), is(newDefinition));
    assertThat(((RecoverableChangeType) change).getAppliedDefinition(), is(definition));
  }

  @Test
  public void updateTemplateKey() {
    PolicyDefinition newDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY_2, API_KEY, newArrayList(), 1, emptyMap());
    ChangeType change = inspector.detectChange(applied(definition), newDefinition);
    assertThat(change, instanceOf(PolicyUpdate.class));
    assertThat(change.getDefinition(), is(newDefinition));
    assertThat(((RecoverableChangeType) change).getAppliedDefinition(), is(definition));
  }

  @Test
  public void updatePointcuts() {
    PolicyDefinition newDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, singletonList(new HttpResourcePointcut("GET", "/foo.*")), 1,
                             emptyMap());
    ChangeType change = inspector.detectChange(applied(definition), newDefinition);
    assertThat(change, instanceOf(PolicyUpdate.class));
    assertThat(change.getDefinition(), is(newDefinition));
    assertThat(((RecoverableChangeType) change).getAppliedDefinition(), is(definition));
  }

  @Test
  public void updateDeploymentFailedApplied() {
    PolicyDefinition anotherDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, newArrayList(), 1, singletonMap("param", "value2"));
    ChangeType change = inspector.detectChange(deploymentFailed(definition, anotherDefinition), differentDefinition);
    assertThat(change, instanceOf(PolicyUpdate.class));
    assertThat(change.getDefinition(), is(differentDefinition));
    assertThat(((RecoverableChangeType) change).getAppliedDefinition(), is(definition));
  }

  @Test
  public void updateDownloadFailedRecoverable() {
    ChangeType change = inspector.detectChange(downloadFailed(definition, differentDefinition), differentDefinition);
    assertThat(change, instanceOf(PolicyUpdate.class));
    assertThat(change.getDefinition(), is(differentDefinition));
    assertThat(((RecoverableChangeType) change).getAppliedDefinition(), is(definition));
  }

  @Test
  public void updateDownloadFailed() {
    ChangeType change = inspector.detectChange(downloadFailed(definition), definition);
    assertThat(change, instanceOf(PolicyAdded.class));
    assertThat(change.getDefinition(), is(definition));
  }

  @Test
  public void updateDeploymentFailed() {
    ChangeType change = inspector.detectChange(deploymentFailed(definition), differentDefinition);
    assertThat(change, instanceOf(PolicyAdded.class));
    assertThat(change.getDefinition(), is(differentDefinition));
  }

  @Test
  public void updateOrderDeploymentFailed() {
    PolicyDefinition reorderedDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, newArrayList(), 2, emptyMap());
    ChangeType change = inspector.detectChange(deploymentFailed(definition), reorderedDefinition);
    assertThat(change, instanceOf(PolicyAdded.class));
    assertThat(change.getDefinition(), is(reorderedDefinition));
  }

  @Test
  public void revertDeploymentFailed() {
    ChangeType change = inspector.detectChange(deploymentFailed(definition, differentDefinition), definition);
    assertThat(change, instanceOf(PolicyRevert.class));
    assertThat(change.getDefinition(), is(definition));
  }

  @Test
  public void revertDownloadFailed() {
    ChangeType change = inspector.detectChange(downloadFailed(definition, differentDefinition), definition);
    assertThat(change, instanceOf(PolicyRevert.class));
    assertThat(change.getDefinition(), is(definition));
  }
}
