/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.store;

import static com.mulesoft.mule.runtime.gw.api.PolicyFolders.getOfflinePoliciesFolder;
import static com.mulesoft.mule.runtime.gw.api.PolicyFolders.getPoliciesFolder;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.CONFIG_DATA;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID_2;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID_3;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.RESOURCE_POINTCUTS;
import static com.mulesoft.mule.runtime.gw.policies.store.DefaultPolicyStore.POLICY_CONFIG_XML_FILE_NAME;
import static com.mulesoft.mule.runtime.gw.policies.store.DefaultPolicyStore.POLICY_DEFINITION_JSON_FILE_NAME;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.io.FileUtils.copyFileToDirectory;
import static org.apache.commons.io.FileUtils.readFileToString;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.hasItems;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_HOME_DIRECTORY_PROPERTY;

import org.mule.runtime.core.api.util.IOUtils;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.SystemPropertyTemporaryFolder;

import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.OfflinePolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.Policy;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.util.List;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class DefaultPolicyStoreTestCase extends AbstractMuleTestCase {

  private static final String VALID_POLICY_CONFIG = "valid XML";
  private static final String UPDATED_VALID_POLICY_CONFIG = "Updated XML";

  @Rule
  public TemporaryFolder muleHome = new SystemPropertyTemporaryFolder(MULE_HOME_DIRECTORY_PROPERTY);

  private PolicyDefinition policyDefinition;
  private PolicyStoreFiles fileHandler;

  private PolicyStore store;

  @Before
  public void setUp() {
    store = new DefaultPolicyStore(new EncryptedPropertiesSerializer());

    fileHandler = new PolicyStoreFiles(getPoliciesFolder());
    PolicyConfiguration policyConfiguration = new PolicyConfiguration(CONFIG_DATA);

    policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, RESOURCE_POINTCUTS, 1, policyConfiguration);
  }

  @Test
  public void validPolicyCorrectlyStored() throws IOException {

    store.store(new Policy(null, policyDefinition, VALID_POLICY_CONFIG));

    File policyFolder = fileHandler.getPolicyFolder(policyDefinition.getName());
    File policyConfigXml = new File(policyFolder, POLICY_CONFIG_XML_FILE_NAME);
    File policyDefinitionJson = new File(policyFolder, POLICY_DEFINITION_JSON_FILE_NAME);
    assertThat(fileHandler.listPolicyFolders(), hasSize(1));
    assertTrue(policyFolder.exists());
    assertTrue(policyConfigXml.exists());
    assertTrue(policyDefinitionJson.exists());
    assertEquals(VALID_POLICY_CONFIG, readFileToString(policyConfigXml));
    assertEquals(IOUtils.getResourceAsString("json/full-definition.json", getClass()).trim(),
                 readFileToString(policyDefinitionJson).trim());
  }

  @Test
  public void containsValidPolicy() {
    store.store(new Policy(null, policyDefinition, VALID_POLICY_CONFIG));

    boolean contains = store.contains(policyDefinition.getName());

    assertTrue(contains);
  }

  @Test
  public void validPolicyCorrectlyDeleted() {
    store.store(new Policy(null, policyDefinition, VALID_POLICY_CONFIG));

    store.remove(policyDefinition.getName());

    File policyFolder = fileHandler.getPolicyFolder(policyDefinition.getName());
    File policyConfigXml = new File(policyFolder, POLICY_CONFIG_XML_FILE_NAME);
    File policyDefinitionJson = new File(policyFolder, POLICY_DEFINITION_JSON_FILE_NAME);
    assertThat(fileHandler.listPolicyFolders(), empty());
    assertFalse(policyFolder.exists());
    assertFalse(policyConfigXml.exists());
    assertFalse(policyDefinitionJson.exists());
  }

  @Test
  public void validPolicyCorrectlyUpdated() throws IOException {
    store.store(new Policy(null, policyDefinition, VALID_POLICY_CONFIG));
    PolicyDefinition updatedPolicyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, null, 2, new PolicyConfiguration(null));

    store.store(new Policy(null, updatedPolicyDefinition, UPDATED_VALID_POLICY_CONFIG));

    File policyFolder = fileHandler.getPolicyFolder(policyDefinition.getName());
    File policyConfigXml = new File(policyFolder, POLICY_CONFIG_XML_FILE_NAME);
    File policyDefinitionJson = new File(policyFolder, POLICY_DEFINITION_JSON_FILE_NAME);
    assertThat(fileHandler.listPolicyFolders(), hasSize(1));
    assertTrue(policyFolder.exists());
    assertTrue(policyConfigXml.exists());
    assertTrue(policyDefinitionJson.exists());
    assertEquals(UPDATED_VALID_POLICY_CONFIG, readFileToString(policyConfigXml));
    assertEquals(IOUtils.getResourceAsString("json/updated-definition.json", getClass()).trim(),
                 readFileToString(policyDefinitionJson).trim());
  }

  @Test
  public void storeDeploymentFailures() {

    store.storeDeploymentFailure(policyDefinition, API_KEY, new RuntimeException("My message exception"));
    store.storeDeploymentFailure(policyDefinition, API_KEY_2, new RuntimeException("My message exception"));

    List<String> fileNames = fileHandler.listPolicyDeploymentFailures().stream().map(File::getName).collect(toList());
    assertThat(fileNames, hasSize(2));
    assertThat(fileNames, hasItems("policyTemplate-policyId@1.log", "policyTemplate-policyId@2.log"));
  }

  @Test
  public void cleanDeploymentFailures() {
    store.storeDeploymentFailure(policyDefinition, API_KEY, new RuntimeException("My message exception"));
    store.storeDeploymentFailure(policyDefinition, API_KEY_2, new RuntimeException("My message exception"));

    store.cleanDeploymentFailure(policyDefinition, API_KEY);

    assertThat(fileHandler.listPolicyDeploymentFailures(), hasSize(1));
    assertThat(fileHandler.listPolicyDeploymentFailures().get(0).getName(), is("policyTemplate-policyId@2.log"));
  }

  @Test
  public void cleanDeploymentFailuresDeletesFolder() {
    store.storeDeploymentFailure(policyDefinition, API_KEY, new RuntimeException("My message exception"));

    store.cleanDeploymentFailure(policyDefinition, API_KEY);

    assertThat(fileHandler.getFailedPoliciesFolder().exists(), is(false));
  }

  @Test
  public void cleanMultipleDeploymentFailures() {
    store.storeDeploymentFailure(policyDefinition, API_KEY, new RuntimeException("My message exception"));
    store.storeDeploymentFailure(policyDefinition, API_KEY_2, new RuntimeException("My message exception"));

    store.cleanDeploymentFailures(policyDefinition.getName());

    assertThat(fileHandler.getFailedPoliciesFolder().exists(), is(false));
  }

  @Test
  public void deleteRemovesFailures() {
    store.storeDeploymentFailure(policyDefinition, API_KEY, new RuntimeException("My message exception"));
    store.store(new Policy(null, policyDefinition, VALID_POLICY_CONFIG));

    store.remove(policyDefinition.getName());

    assertThat(fileHandler.listPolicyFolders(), empty());
    assertThat(fileHandler.getFailedPoliciesFolder().exists(), is(false));
  }

  @Test
  public void loadAll() {
    PolicyDefinition policyDefinition2 =
        new PolicyDefinition(POLICY_ID_2, POLICY_TEMPLATE_KEY, API_KEY, null, 2, new PolicyConfiguration(null));
    store.store(new Policy(null, policyDefinition, VALID_POLICY_CONFIG));
    store.store(new Policy(null, policyDefinition2, VALID_POLICY_CONFIG));

    List<PolicyDefinition> allPolicies = store.load();

    assertThat(allPolicies, hasSize(2));
    assertTrue(allPolicies.contains(policyDefinition));
    assertTrue(allPolicies.contains(policyDefinition2));
  }

  @Test
  public void onlinePolicies() {
    PolicyDefinition policyDefinition2 =
        new PolicyDefinition(POLICY_ID_2, POLICY_TEMPLATE_KEY, API_KEY, null, 1, new PolicyConfiguration(null));
    PolicyDefinition offlinePolicy =
        new OfflinePolicyDefinition(POLICY_ID_3, POLICY_TEMPLATE_KEY, API_KEY, null, 1, new PolicyConfiguration(null));
    store.store(new Policy(null, policyDefinition, VALID_POLICY_CONFIG));
    store.store(new Policy(null, policyDefinition2, VALID_POLICY_CONFIG));
    store.store(new Policy(null, offlinePolicy, VALID_POLICY_CONFIG));

    List<PolicyDefinition> onlinePolicies = store.onlinePolicies();

    assertThat(onlinePolicies, hasSize(2));
    assertThat(onlinePolicies, hasItems(policyDefinition, policyDefinition2));
  }

  @Test
  public void offlinePolicies() throws URISyntaxException, IOException {
    store.store(new Policy(null, policyDefinition, VALID_POLICY_CONFIG));
    copyFileToDirectory(definitionFile(), getOfflinePoliciesFolder());
    copyFileToDirectory(anotherDefinitionFile(), getOfflinePoliciesFolder());
    copyFileToDirectory(invalidDefinitionFile(), getOfflinePoliciesFolder());

    List<PolicyDefinition> offlinePolicies = store.offlinePolicies();

    List<String> policyNames = offlinePolicies.stream().map(PolicyDefinition::getName).collect(toList());
    assertThat(offlinePolicies, hasSize(2));
    assertThat(policyNames, hasItems("offline-multiple-apis-definition", "offline-single-api-definition"));
  }

  private File definitionFile() throws URISyntaxException {
    return new File(getClass().getResource("/json/offline-single-api-definition.json").toURI());
  }

  private File anotherDefinitionFile() throws URISyntaxException {
    return new File(getClass().getResource("/json/offline-multiple-apis-definition.json").toURI());
  }

  private File invalidDefinitionFile() throws URISyntaxException {
    return new File(getClass().getResource("/json/offline-invalid-field-definition.json").toURI());
  }

}
