/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.template;

import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static java.util.Arrays.asList;
import static org.custommonkey.xmlunit.XMLAssert.assertXMLEqual;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mule.runtime.core.api.util.IOUtils.getResourceAsString;

import org.mule.tck.junit4.AbstractMuleTestCase;

import com.mulesoft.mule.runtime.gw.policies.template.exception.PolicyTemplateResolverException;
import com.mulesoft.mule.runtime.gw.policies.template.provider.PolicyTemplateProvider;
import com.mulesoft.mule.runtime.gw.policies.template.resolver.HandlebarsPolicyTemplateResolver;
import com.mulesoft.mule.runtime.gw.policies.template.resolver.PolicyTemplateResolver;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.custommonkey.xmlunit.XMLUnit;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;
import org.xml.sax.SAXException;

public class HandlebarsPolicyTemplateResolverTestCase extends AbstractMuleTestCase {

  private static final Item ITEM = new Item("Item 1", "$19.99", asList(new Feature("New!"), new Feature("Awesome!")));
  private static final Item ITEM_2 = new Item("Item 2", "$29.99", asList(new Feature("Old."), new Feature("Ugly.")));

  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  @Rule
  public ExpectedException expectedException = ExpectedException.none();

  private Map<String, Object> configurationData;
  private PolicyTemplate template;

  private PolicyTemplateResolver policyTemplateResolver;

  @Before
  public void setUp() throws IOException, URISyntaxException {
    configurationData = new HashMap<>();

    File templateFile = new File(getClass().getResource("/templates/template.xml").getFile());

    template = mock(PolicyTemplate.class);
    when(template.getTemplateFile()).thenReturn(templateFile);
    when(template.getKey()).thenReturn(POLICY_TEMPLATE_KEY);

    PolicyTemplateProvider templateProvider = mock(PolicyTemplateProvider.class);
    when(templateProvider.provide(POLICY_TEMPLATE_KEY)).thenReturn(template);
    policyTemplateResolver = new HandlebarsPolicyTemplateResolver();

    XMLUnit.setIgnoreWhitespace(true);
  }

  @Test
  public void resolveTemplate() throws IOException, PolicyTemplateResolverException {
    configurationData.put("items", asList(ITEM, ITEM_2));
    configurationData.put("boolean", true);

    String resolvedTemplate = policyTemplateResolver.resolve(template, configurationData);

    assertThat(resolvedTemplate, is(getResourceAsString("templates/resolved-template.xml", getClass())));
  }

  @Test
  public void resolveTemplateWithEqualsHelperA() throws IOException, PolicyTemplateResolverException, SAXException {
    File templateFile = new File(getClass().getResource("/templates/template-with-equals-helper.xml").getFile());
    when(template.getTemplateFile()).thenReturn(templateFile);
    configurationData.put("value", "A");

    String resolvedTemplate = policyTemplateResolver.resolve(template, configurationData);

    assertXMLEqual(resolvedTemplate,
                   getResourceAsString("templates/resolved-template-with-equals-helper-a.xml", getClass()));
  }

  @Test
  public void resolveTemplateWithEqualsHelperB() throws IOException, PolicyTemplateResolverException, SAXException {
    File templateFile = new File(getClass().getResource("/templates/template-with-equals-helper.xml").getFile());
    when(template.getTemplateFile()).thenReturn(templateFile);
    configurationData.put("value", "B");

    String resolvedTemplate = policyTemplateResolver.resolve(template, configurationData);

    assertXMLEqual(resolvedTemplate,
                   getResourceAsString("templates/resolved-template-with-equals-helper-b.xml", getClass()));
  }

  @Test
  public void resolveTemplateWithEqualsHelperC() throws IOException, PolicyTemplateResolverException, SAXException {
    File templateFile = new File(getClass().getResource("/templates/template-with-equals-helper.xml").getFile());
    when(template.getTemplateFile()).thenReturn(templateFile);
    configurationData.put("value", "C");

    String resolvedTemplate = policyTemplateResolver.resolve(template, configurationData);

    assertXMLEqual(resolvedTemplate,
                   getResourceAsString("templates/resolved-template-with-equals-helper-c.xml", getClass()));
  }

  @Test
  public void parseTemplateWithUnusedScope() throws IOException, PolicyTemplateResolverException {
    configurationData.put("items", asList(ITEM, ITEM_2));
    configurationData.put("unusedScope", "value");
    configurationData.put("boolean", true);

    String resolvedTemplate = policyTemplateResolver.resolve(template, configurationData);

    assertThat(resolvedTemplate, is(getResourceAsString("templates/resolved-template.xml", getClass())));
  }

  @Test
  public void parseTemplateWithTemplateVarsNotEvaluated() throws IOException, PolicyTemplateResolverException {
    configurationData.put("unusedScope", "value");

    String resolvedTemplate = policyTemplateResolver.resolve(template, configurationData);

    assertThat(resolvedTemplate, is(getResourceAsString("templates/empty-template.xml", getClass())));
  }

  static class Item {

    private List<Feature> features;
    private String name;
    private String price;

    Item(String name, String price, List<Feature> features) {
      this.name = name;
      this.price = price;
      this.features = features;
    }

    public String getName() {
      return name;
    }

    public String getPrice() {
      return price;
    }

    public List<Feature> getFeatures() {
      return features;
    }
  }

  static class Feature {

    private String description;

    Feature(String description) {
      this.description = description;
    }

    public String getDescription() {
      return description;
    }
  }

}
