/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.policies.factory;

import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.API_KEY;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_ID;
import static com.mulesoft.anypoint.tests.PolicyTestValuesConstants.POLICY_TEMPLATE_KEY;
import static com.mulesoft.mule.runtime.gw.internal.encryption.RuntimeEncrypterFactory.createDefaultRuntimeEncrypter;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.empty;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static java.util.Arrays.asList;

import org.mule.runtime.deployment.model.api.policy.PolicyTemplateDescriptor;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.junit4.rule.SystemProperty;

import com.mulesoft.mule.runtime.gw.model.IdentityManagement;
import com.mulesoft.mule.runtime.gw.model.PolicyConfiguration;
import com.mulesoft.mule.runtime.gw.model.PolicyDefinition;
import com.mulesoft.mule.runtime.gw.model.PolicyProperty;
import com.mulesoft.mule.runtime.gw.model.PolicySpecification;
import com.mulesoft.mule.runtime.gw.policies.OfflinePolicyDefinition;
import com.mulesoft.mule.runtime.gw.policies.Policy;
import com.mulesoft.mule.runtime.gw.policies.encryption.DefaultPolicyConfigurationEncrypter;
import com.mulesoft.mule.runtime.gw.policies.encryption.PolicyConfigurationEncrypter;
import com.mulesoft.mule.runtime.gw.policies.encryption.PolicyConfigurationEncryptionResult;
import com.mulesoft.mule.runtime.gw.policies.template.PolicyTemplate;
import com.mulesoft.mule.runtime.gw.policies.template.provider.PolicyTemplateProvider;
import com.mulesoft.mule.runtime.gw.policies.template.resolver.PolicyTemplateResolver;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;

import java.io.File;
import java.util.Map;
import java.util.Optional;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class EncryptedPolicyFactoryTestCase extends AbstractMuleTestCase {

  private static final Map<String, Object> CONFIG_DATA = ImmutableMap.of("key", "some value");
  private static final Map<String, Object> IDENTITY_MANAGEMENT_CONFIG_DATA = ImmutableMap.of("key", "some value",
                                                                                             "identityManagementTokenUrl",
                                                                                             "identityManagementTokenUrlValue",
                                                                                             "identityManagementClientSecret",
                                                                                             "u6gMLNLHDv4QRAqqzRezPqcDoohC0vMTvtKUqy5",
                                                                                             "identityManagementClientId",
                                                                                             "api_platform");

  private static final Map<String, Object> PLACEHOLDERS = ImmutableMap.of("key", "${secure::key}");
  private static final Map<String, Object> ENCRYPTED_PLACEHOLDERS = ImmutableMap.of("encrypted", true, "key", "${secure::key}");
  private static final Map<String, Object> ENCRYPTED_IDENTITY_MANAGEMENT_PLACEHOLDERS =
      ImmutableMap.of("encrypted", true, "key", "${secure::key}", "identityManagementTokenUrl",
                      "${secure::identityManagementTokenUrl}",
                      "identityManagementClientSecret",
                      "${secure::identityManagementClientSecret}",
                      "identityManagementClientId",
                      "${secure::identityManagementClientId}");

  private static final Map<String, Object> ENCRYPTED_CONFIG_DATA_MOCK = Maps.newHashMap(ImmutableMap.of("key", "encryptedValue"));
  private static final Map<String, Object> ENCRYPTED_CONFIG_DATA =
      Maps.newHashMap(ImmutableMap.of("encrypted", true, "key", "![cxDrfb5eezub4FgbvbHyHg==]"));
  private static final Map<String, Object> ENCRYPTED_IDENTITY_MANAGEMENT_CONFIG_DATA =
      Maps.newHashMap(ImmutableMap.of("encrypted", true, "key", "![cxDrfb5eezub4FgbvbHyHg==]",
                                      "identityManagementTokenUrl", "![rQ1yMhp8i/g3ZUD+Ojnym4HaYt1AMEi1DFEcBeU9+wQ=]",
                                      "identityManagementClientSecret",
                                      "![7PzjHgPi/HLFgoJkamHv4kz8AcZhTah3GA30HfMftqlqvtSdq1pMgb7yD6fSw/iL]",
                                      "identityManagementClientId", "![fHLAAKD9KfOFL0ylTUZmww==]"));

  private PolicyFactory policyFactory;

  @Mock
  private PolicyConfigurationEncrypter encrypter;

  @Mock
  private PolicySpecification policySpecification;

  private PolicyTemplate template;

  @Mock
  private PolicyTemplateResolver templateResolver;
  @Mock
  private PolicyTemplateProvider templateProvider;

  private String resolvedTemplate = "resolvedTemplate";
  private String resolvedNotEncryptedTemplate = "resolvedNotEncryptedTemplate";

  private static final String ENCRYPTION_KEY = "GatewayTeamKey00";
  @Rule
  public SystemProperty encryptionKey = new SystemProperty("anypoint.platform.encryption_key", ENCRYPTION_KEY);

  @Before
  public void setUp() {
    template = new PolicyTemplate(POLICY_TEMPLATE_KEY, mock(File.class), policySpecification,
                                  new PolicyTemplateDescriptor(POLICY_TEMPLATE_KEY.getName()));

    when(templateProvider.provide(any())).thenReturn(template);
    when(templateResolver.resolve(template, PLACEHOLDERS)).thenReturn(resolvedTemplate);
    when(templateResolver.resolve(template, ENCRYPTED_PLACEHOLDERS)).thenReturn(resolvedTemplate);
    when(templateResolver.resolve(template, ENCRYPTED_IDENTITY_MANAGEMENT_PLACEHOLDERS)).thenReturn(resolvedTemplate);
    when(templateResolver.resolve(template, CONFIG_DATA)).thenReturn(resolvedNotEncryptedTemplate);

    policyFactory = new EncryptedPolicyFactory(templateResolver, templateProvider, encrypter);
  }

  @Test
  public void createPolicy() {
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, null, 1, new PolicyConfiguration(CONFIG_DATA));

    when(encrypter.encryptConfiguration(eq(policyDefinition), any()))
        .thenReturn(new PolicyConfigurationEncryptionResult(PLACEHOLDERS, new PolicyConfiguration(ENCRYPTED_CONFIG_DATA_MOCK),
                                                            null));

    when(policySpecification.isEncryptionSupported()).thenReturn(true);

    Policy policy = policyFactory.createFromPolicyDefinition(policyDefinition);

    assertThat(policy.getPolicyTemplate(), is(template));
    assertThat(policy.getPolicyDefinition().getId(), is(POLICY_ID));
    assertThat(policy.getPolicyDefinition().getTemplateKey(), is(POLICY_TEMPLATE_KEY));
    assertThat(policy.getPolicyDefinition().getApiKeys(), contains(API_KEY));
    assertThat(policy.getPolicyDefinition().getOrder(), is(1));
    assertThat(policy.getPolicyDefinition().getHttpResourcePointcuts(), empty());
    assertThat(policy.getPolicyDefinition().isOnline(), is(true));
    assertThat(policy.getPolicyDefinition().getConfigurationData().getConfiguration(), is(ENCRYPTED_CONFIG_DATA_MOCK));
    assertThat(policy.getResolvedTemplate(), is(resolvedTemplate));
  }

  @Test
  public void createPolicyRemovingIdentityManagement() {
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, null, 1,
                             new PolicyConfiguration(IDENTITY_MANAGEMENT_CONFIG_DATA));

    PolicyConfigurationEncrypter policyConfigurationEncrypter =
        new DefaultPolicyConfigurationEncrypter(createDefaultRuntimeEncrypter(), false);


    when(policySpecification.isEncryptionSupported()).thenReturn(true);
    when(policySpecification.isValid()).thenReturn(true);
    when(policySpecification.getIdentityManagement()).thenReturn(Optional.empty());
    PolicyProperty policyProperty = new PolicyProperty();
    policyProperty.setType("string");
    policyProperty.setPropertyName("key");
    when(policySpecification.getConfiguration()).thenReturn(asList(policyProperty));

    PolicyFactory encryptedPolicyFactory =
        new EncryptedPolicyFactory(templateResolver, templateProvider, policyConfigurationEncrypter);

    Policy policy = encryptedPolicyFactory.createFromPolicyDefinition(policyDefinition);

    assertThat(policy.getPolicyTemplate(), is(template));
    assertThat(policy.getPolicyDefinition().getId(), is(POLICY_ID));
    assertThat(policy.getPolicyDefinition().getTemplateKey(), is(POLICY_TEMPLATE_KEY));
    assertThat(policy.getPolicyDefinition().getApiKeys(), contains(API_KEY));
    assertThat(policy.getPolicyDefinition().getOrder(), is(1));
    assertThat(policy.getPolicyDefinition().getHttpResourcePointcuts(), empty());
    assertThat(policy.getPolicyDefinition().isOnline(), is(true));
    assertThat(policy.getPolicyDefinition().getConfigurationData().getConfiguration(),
               is(ENCRYPTED_CONFIG_DATA));
    assertThat(policy.getResolvedTemplate(), is(resolvedTemplate));
  }

  @Test
  public void createPolicyNotRemovingIdentityManagement() {
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, null, 1,
                             new PolicyConfiguration(IDENTITY_MANAGEMENT_CONFIG_DATA));

    PolicyConfigurationEncrypter policyConfigurationEncrypter =
        new DefaultPolicyConfigurationEncrypter(createDefaultRuntimeEncrypter(), false);


    when(policySpecification.isEncryptionSupported()).thenReturn(true);
    when(policySpecification.isValid()).thenReturn(true);
    when(policySpecification.getIdentityManagement()).thenReturn(Optional.of(mock(IdentityManagement.class)));
    PolicyProperty policyProperty = new PolicyProperty();
    policyProperty.setType("string");
    policyProperty.setPropertyName("key");
    when(policySpecification.getConfiguration()).thenReturn(asList(policyProperty));

    PolicyFactory encryptedPolicyFactory =
        new EncryptedPolicyFactory(templateResolver, templateProvider, policyConfigurationEncrypter);

    Policy policy = encryptedPolicyFactory.createFromPolicyDefinition(policyDefinition);

    assertThat(policy.getPolicyTemplate(), is(template));
    assertThat(policy.getPolicyDefinition().getId(), is(POLICY_ID));
    assertThat(policy.getPolicyDefinition().getTemplateKey(), is(POLICY_TEMPLATE_KEY));
    assertThat(policy.getPolicyDefinition().getApiKeys(), contains(API_KEY));
    assertThat(policy.getPolicyDefinition().getOrder(), is(1));
    assertThat(policy.getPolicyDefinition().getHttpResourcePointcuts(), empty());
    assertThat(policy.getPolicyDefinition().isOnline(), is(true));
    assertThat(policy.getPolicyDefinition().getConfigurationData().getConfiguration(),
               is(ENCRYPTED_IDENTITY_MANAGEMENT_CONFIG_DATA));
    assertThat(policy.getResolvedTemplate(), is(resolvedTemplate));
  }

  @Test
  public void createPolicyNotEncryptionEnabled() {
    PolicyDefinition policyDefinition =
        new PolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, null, 1, new PolicyConfiguration(CONFIG_DATA));

    when(policySpecification.isEncryptionSupported()).thenReturn(false);

    Policy policy = policyFactory.createFromPolicyDefinition(policyDefinition);

    assertThat(policy.getPolicyTemplate(), is(template));
    assertThat(policy.getPolicyDefinition().getId(), is(POLICY_ID));
    assertThat(policy.getPolicyDefinition().getTemplateKey(), is(POLICY_TEMPLATE_KEY));
    assertThat(policy.getPolicyDefinition().getApiKeys(), contains(API_KEY));
    assertThat(policy.getPolicyDefinition().getOrder(), is(1));
    assertThat(policy.getPolicyDefinition().getHttpResourcePointcuts(), empty());
    assertThat(policy.getPolicyDefinition().isOnline(), is(true));
    assertThat(policy.getPolicyDefinition().getConfigurationData().getConfiguration(), is(CONFIG_DATA));
    assertThat(policy.getResolvedTemplate(), is(resolvedNotEncryptedTemplate));
  }

  @Test
  public void createOfflinePolicy() {
    PolicyDefinition policyDefinition =
        new OfflinePolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, null, 1, new PolicyConfiguration(CONFIG_DATA));

    when(encrypter.encryptConfiguration(eq(policyDefinition), any()))
        .thenReturn(new PolicyConfigurationEncryptionResult(PLACEHOLDERS, new PolicyConfiguration(ENCRYPTED_CONFIG_DATA_MOCK),
                                                            null));

    when(policySpecification.isEncryptionSupported()).thenReturn(true);

    Policy policy = policyFactory.createFromPolicyDefinition(policyDefinition);

    assertThat(policy.getPolicyTemplate(), is(template));
    assertThat(policy.getPolicyDefinition().getId(), is(POLICY_ID));
    assertThat(policy.getPolicyDefinition().getTemplateKey(), is(POLICY_TEMPLATE_KEY));
    assertThat(policy.getPolicyDefinition().getApiKeys(), contains(API_KEY));
    assertThat(policy.getPolicyDefinition().getOrder(), is(1));
    assertThat(policy.getPolicyDefinition().getHttpResourcePointcuts(), empty());
    assertThat(policy.getPolicyDefinition().isOnline(), is(false));
    assertThat(policy.getPolicyDefinition().getConfigurationData().getConfiguration(), is(ENCRYPTED_CONFIG_DATA_MOCK));
    assertThat(policy.getResolvedTemplate(), is(resolvedTemplate));
  }

  @Test
  public void createOfflinePolicyNotEncryptionEnabled() {
    PolicyDefinition policyDefinition =
        new OfflinePolicyDefinition(POLICY_ID, POLICY_TEMPLATE_KEY, API_KEY, null, 1, new PolicyConfiguration(CONFIG_DATA));

    when(policySpecification.isEncryptionSupported()).thenReturn(false);

    Policy policy = policyFactory.createFromPolicyDefinition(policyDefinition);

    assertThat(policy.getPolicyTemplate(), is(template));
    assertThat(policy.getPolicyDefinition().getId(), is(POLICY_ID));
    assertThat(policy.getPolicyDefinition().getTemplateKey(), is(POLICY_TEMPLATE_KEY));
    assertThat(policy.getPolicyDefinition().getApiKeys(), contains(API_KEY));
    assertThat(policy.getPolicyDefinition().getOrder(), is(1));
    assertThat(policy.getPolicyDefinition().getHttpResourcePointcuts(), empty());
    assertThat(policy.getPolicyDefinition().isOnline(), is(false));
    assertThat(policy.getPolicyDefinition().getConfigurationData().getConfiguration(), is(CONFIG_DATA));
    assertThat(policy.getResolvedTemplate(), is(resolvedNotEncryptedTemplate));
  }

}
