/*
 * Copyright 2024 Salesforce, Inc. All rights reserved.
 */
package com.mulesoft.anypoint;


import static com.mulesoft.anypoint.model.policy.VersionParser.parseVersions;

import com.mulesoft.anypoint.model.policy.definition.Definition;
import com.mulesoft.anypoint.model.policy.implementation.Implementation;
import com.mulesoft.anypoint.model.policy.implementation.ImplementationBuilder;
import com.mulesoft.anypoint.model.policy.implementation.MuleArtifact;
import com.mulesoft.anypoint.model.serde.PolicyMapper;

import java.io.File;
import java.util.List;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;

/**
 * Builds the policy implementations files.
 */
@Mojo(name = "build-asset-files", defaultPhase = LifecyclePhase.COMPILE)
public class BuildAssetFilesMojo extends AbstractMojo {

  /**
   * Location of the policy definition.
   */
  @Parameter(defaultValue = "definition/target", property = "definitionDirectory", required = true)
  private File definitionDirectory;

  /**
   * Location of the policy implementation.
   */
  @Parameter(defaultValue = "${project.basedir}", property = "implementationDirectory", required = true)
  private File implementationDirectory;

  /**
   * Location where the output will be sent.
   */
  @Parameter(defaultValue = "${project.build.directory}", property = "outputDir", required = true)
  private File outputDirectory;

  /**
   * groupId of the project.
   */
  @Parameter(defaultValue = "${project.groupId}", property = "groupId", required = true)
  private String groupId;

  /**
   * artifactId of the project.
   */
  @Parameter(defaultValue = "${project.artifactId}", property = "artifactId", required = true)
  private String artifactId;

  /**
   * Version of the project.
   */
  @Parameter(defaultValue = "${project.version}", property = "version", required = true)
  private String version;

  /**
   * Name of the project.
   */
  @Parameter(defaultValue = "${project.name}", property = "name", required = true)
  private String name;

  /**
   * Supported java versions for the policy.
   */
  @Parameter(defaultValue = "${supportedJavaVersions}", property = "supportedJavaVersions")
  private String supportedJavaVersions;

  private PolicyMapper policyMapper = new PolicyMapper();

  public void execute() throws MojoExecutionException {
    Definition def = policyMapper.parseDefinitionFiles(definitionDirectory);
    MuleArtifact muleArtifact = policyMapper.parseMuleArtifact(implementationDirectory);
    List<String> javaVersions = parseVersions(supportedJavaVersions);

    Implementation impl = new ImplementationBuilder()
        .withDefinition(def)
        .withMuleArtifact(muleArtifact)
        .withGav(groupId, artifactId, version)
        .withName(name)
        .withJavaVersions(javaVersions)
        .build();

    policyMapper.writeImplementationFiles(outputDirectory, impl);
  }
}
