/*
 * Copyright 2024 Salesforce, Inc. All rights reserved.
 */
package com.mulesoft.anypoint.model.policy.implementation;

import static java.lang.String.format;
import static java.util.Collections.singletonList;

import com.mulesoft.anypoint.model.policy.definition.Definition;
import com.mulesoft.anypoint.model.policy.exchange.Dependency;
import com.mulesoft.anypoint.model.policy.exchange.Exchange;

import java.util.List;

import org.apache.maven.plugin.MojoExecutionException;

public class ImplementationBuilder {

  public static final String TECHNOLOGY = "mule4";
  public static final String MAIN = null; // TODO W-17219086 set "gcl.yaml" once it is properly generated.
  public static final String POSFIX = "-mule";

  private Definition definition;
  private String groupId;
  private String artifactId;
  private String version;
  private String name;
  private MuleArtifact muleArtifact;
  private List<String> supportedJavaVersions;

  public ImplementationBuilder() {}

  public ImplementationBuilder withDefinition(Definition definition) {
    this.definition = definition;
    return this;
  }

  public ImplementationBuilder withGav(String groupId, String artifactId, String version) {
    this.groupId = groupId;
    this.artifactId = artifactId;
    this.version = version;
    return this;
  }

  public ImplementationBuilder withName(String name) {
    this.name = name;
    return this;
  }

  public ImplementationBuilder withMuleArtifact(MuleArtifact muleArtifact) {
    this.muleArtifact = muleArtifact;
    return this;
  }

  public Implementation build() throws MojoExecutionException {
    validateGavMatchesDefinition();
    Exchange exchange = buildImplExchange();
    ImplementationMetadata metadata = buildMetadata();

    return new Implementation(exchange, metadata);
  }

  private void validateGavMatchesDefinition() throws MojoExecutionException {
    Exchange exchange = definition.getExchange();
    if (!groupId.equals(exchange.groupId)) {
      throw new MojoExecutionException(format("Implementation groupId %s does not match with definition groupId %s.", groupId,
                                              exchange.groupId));
    }

    if (!artifactId.equals(exchange.assetId + POSFIX)) {
      throw new MojoExecutionException(format("Implementation artifactId does not match with the pattern <definition-assetId>-mule. Expected: %s%s, Actual: %s.",
                                              exchange.assetId, POSFIX, artifactId));
    }
  }

  private Exchange buildImplExchange() {
    Exchange defExchange = definition.getExchange();
    Dependency exchangeDep = new Dependency(defExchange.groupId, defExchange.assetId, defExchange.version);
    return new Exchange(MAIN, groupId, artifactId, version, name, singletonList(exchangeDep));
  }

  private ImplementationMetadata buildMetadata() {
    return new ImplementationMetadata(name, TECHNOLOGY, muleArtifact.minMuleVersion, supportedJavaVersions);
  }

  public ImplementationBuilder withJavaVersions(List<String> supportedJavaVersions) {
    this.supportedJavaVersions = supportedJavaVersions;
    return this;
  }
}
