/*
 * Copyright 2024 Salesforce, Inc. All rights reserved.
 */
package com.mulesoft.anypoint.model.serde;

import static java.lang.String.format;

import com.mulesoft.anypoint.model.policy.definition.DefMetadata;
import com.mulesoft.anypoint.model.policy.definition.Definition;
import com.mulesoft.anypoint.model.policy.definition.DefinitionGcl;
import com.mulesoft.anypoint.model.policy.exchange.Exchange;
import com.mulesoft.anypoint.model.policy.implementation.Implementation;
import com.mulesoft.anypoint.model.policy.implementation.ImplementationMetadata;
import com.mulesoft.anypoint.model.policy.implementation.MuleArtifact;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import com.fasterxml.jackson.dataformat.yaml.YAMLGenerator;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;

import org.apache.maven.plugin.MojoExecutionException;

public class PolicyMapper {

  private static final String EXCHANGE_JSON = "exchange.json";
  private static final String GCL_YAML = "gcl.yaml";
  private static final String METADATA_YAML = "metadata.yaml";
  private static final String MULE_ARTIFACT = "mule-artifact.json";
  private static final String METADATA_YAML_HEADER = "#%Policy Implementation 1.0\n";

  private final ObjectMapper jsonMapper = new ObjectMapper()
      .setSerializationInclusion(JsonInclude.Include.NON_NULL)
      .enable(SerializationFeature.INDENT_OUTPUT)
      .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);

  private final ObjectMapper yamlMapper =
      new ObjectMapper(new YAMLFactory().disable(YAMLGenerator.Feature.WRITE_DOC_START_MARKER))
          .setSerializationInclusion(JsonInclude.Include.NON_NULL)
          .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);


  public Definition parseDefinitionFiles(File definitionDirectory) throws MojoExecutionException {
    if (!definitionDirectory.exists()) {
      throw new MojoExecutionException("Definition directory does not exist. Directory: "
          + definitionDirectory.getAbsolutePath());
    }
    Exchange exchange = readFile(definitionDirectory, EXCHANGE_JSON, jsonMapper, Exchange.class);
    DefinitionGcl gcl = readFile(definitionDirectory, GCL_YAML, yamlMapper, DefinitionGcl.class);
    DefMetadata metadata = readFile(definitionDirectory, METADATA_YAML, yamlMapper, DefMetadata.class);

    return new Definition(exchange, gcl, metadata);
  }

  public MuleArtifact parseMuleArtifact(File location) throws MojoExecutionException {
    return readFile(location, MULE_ARTIFACT, jsonMapper, MuleArtifact.class);
  }

  public void writeImplementationFiles(File outputDirectory, Implementation impl) throws MojoExecutionException {
    setupOutputDirectory(outputDirectory);

    writeExchange(outputDirectory, impl.getExchange());
    writeMetadata(outputDirectory, impl.getMetadata());
  }

  private <E> E readFile(File directory, String name, ObjectMapper mapper, Class<E> clazz)
      throws MojoExecutionException {
    File file = new File(directory, name);
    if (!file.exists()) {
      throw new MojoExecutionException(format("File %s missing from directory %s.", name, directory.getAbsolutePath()));
    }
    try {
      return mapper.readValue(file, clazz);
    } catch (IOException e) {
      throw new MojoExecutionException(format("File %s could not be parsed.", name), e);
    }
  }

  private void writeExchange(File outputDirectory, Exchange exchange) throws MojoExecutionException {
    File file = new File(outputDirectory, EXCHANGE_JSON);
    try {
      jsonMapper.writeValue(file, exchange);
    } catch (IOException e) {
      throw new MojoExecutionException(format("File %s could not be written.", EXCHANGE_JSON), e);
    }
  }

  private void writeMetadata(File outputDirectory, ImplementationMetadata metadata) throws MojoExecutionException {
    try {
      FileWriter writer = new FileWriter(new File(outputDirectory, METADATA_YAML));
      writer.write(METADATA_YAML_HEADER);
      yamlMapper.writeValue(writer, metadata);
      writer.close();
    } catch (IOException e) {
      throw new MojoExecutionException(format("File %s could not be written.", METADATA_YAML), e);
    }
  }

  private void setupOutputDirectory(File outputDirectory) throws MojoExecutionException {
    if (!outputDirectory.exists()) {
      if (!outputDirectory.mkdirs()) {
        throw new MojoExecutionException("Error creating output directory: " + outputDirectory.getAbsolutePath());
      }
    }
  }


}
