%dw 2.0
import envVar from dw::System
import prop from dw::Runtime

/**
* Extracts the given "key" from the giver reporter options or from an env variable if not found there
* Will return and empty string if not found and log a message
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | options | the current reporter options
* | key | the key of the options map y
* |===
*
* === Example
*
* This example shows how the `getReporterProperty` behaves under different inputs.
*
* ==== Options
*
* [source,Yaml,linenums]
* ----
*   - type: Email
*       options:
*         EMAILS: "nbattiato@example.org"
* ----
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* import * from bat::Reporters::Common
* var myOption = getReporterProperty(options, "EMAILS")
* ---
* myOption
*
* ----
*
* ==== Output
*
* [source,Text,linenums]
* ----
* "nbattiato@example.org"
* ----
**/
fun getReporterProperty(options: Object | Null, key: String): Any =
  if(options != null and options."$key"?)
    options."$key"
  else if((dw::System::envVar(key) default '') != '')
    dw::System::envVar(key)
  else
    log("Missing $key env variable or option", '')

/**
*
* Checks if there is a DISABLED_FOR_CONFIGS property in the reporter options.
* If that property is specified it will check if the current config is one of the DISABLED_FOR_CONFIGS
* Returns true if the current config is in the DISABLED_FOR_CONFIGS, or false if it isn't
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | options | the current reporter options
* | extraContext | the extra config map, at the moment only a config property will be there
* |===
*
* === Example
*
* This example shows how the `isDisabled` behaves under different inputs. Assuming the current config is `devx`.
*
* ==== Options
*
* [source,Yaml,linenums]
* ----
*   - type: Email
*       options:
*         EMAILS: "nbattiato@example.org"
*         DISABLED_FOR_CONFIGS: "devx,qax"
* ----
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* import * from bat::Reporters::Common
* ---
* isDisabled(options, extraConfig)
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* true
* ----
**/
fun isDisabled(options: Object | Null, extraContext: Object): Boolean = do {
    var configsDisabled: Array<String> = do {
        var configs: String  = getReporterProperty(options, "DISABLED_FOR_CONFIGS") as String
        ---
        if(configs != null and configs != '')
           if(configs contains(","))
               configs splitBy(",")
           else if(configs contains(";"))
               configs splitBy(";")
           else
               [configs]
        else
          log('Missing DISABLED_FOR_CONFIGS option', [""])
    }

    var currentConfig: String =
        if (extraContext != null and extraContext.config?)
            extraContext.config as String
        else
          log('Missing config value in extraContext', "")
    ---
    if(currentConfig != null and currentConfig != "")
        configsDisabled contains currentConfig
    else
        false
}
/**
* Checks if there is a SEND_NOTIFICATION_IF_PASSED property in the reporter options.
*
* And returns it's boolean value, or false if its not a boolean or a string like 'true'
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | options | the current reporter options
* |===
*
* === Example
*
* This example shows how the `sendNotificationIfPassed` behaves under different inputs.
*
* ==== Options
*
* [source,Yaml,linenums]
* ----
*   - type: Email
*       options:
*         EMAILS: "nbattiato@example.org"
*         SEND_NOTIFICATION_IF_PASSED: 'true' // you can also use a boolean
* ----
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* import * from bat::Reporters::Common
* ---
* sendNotifications(options)
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* true
* ----
**/
fun sendNotificationIfPassed(options: Object | Null): Boolean  = getBooleanProperty(options, "SEND_NOTIFICATION_IF_PASSED")
/**
* Gets the specified boolean property from reporter options if specified a valid value.
*
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | options | current reporter options
* | property | name of the property
* |===
*
* === Example
*
* This example shows how the `getBooleanProperty` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* getBooleanProperty("IS_ENABLED")  // assuming IS_ENABLES is defined, and is a boolean or the string 'true'
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* true
* ----
**/
fun getBooleanProperty(options: Object | Null, property: String): Boolean  = do {
 var propertyValue: Any = getReporterProperty(options, property)
 ---
 if(propertyValue is Boolean)
    propertyValue as Boolean
 else if(propertyValue is String and truthy(propertyValue as String))
    true
 else
   false
}

/**
* Returns the boolean representation of string
* for example 'true' returns the boolean `true`. any other string or null returns the boolean `false`
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | value |
* |===
*
* === Example
*
* This example shows how the `truthy` behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
* truthy('true')
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* true
* ----
**/
fun truthy(value: String | Null) = value default '' matches /true/


//get host if target is mule
var anypoint_host_env = (envVar('anypoint-host'))
var anypoint_host_prop = (prop('anypoint-host'))

//get host if target is lambda
var BAT_ENV_ENV = (envVar('BAT_ENV'))
var BAT_ENV_PROP = (prop('BAT_ENV'))


/**
* Returns the url to the monitor for the given payload
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | payload | bat result payload
* |===
**/
fun generateLink(payload) = 'https://'++ baseUri() ++'/monitoring/#/apitesting/ui/suite/' ++ splitBy((payload.metadata.testId),':')[0] ++':'++ splitBy((payload.metadata.testId),':')[1]
                             ++ '/version/' ++ splitBy((payload.metadata.testId),':')[2] ++ "/execution/" ++ payload.metadata.executionId

/**
* Returns the Anypoint base url
**/
fun baseUri() =  if(anypoint_host_env != null) anypoint_host_env
                             else if (anypoint_host_prop != null) anypoint_host_prop
                             else if (BAT_ENV_ENV != null) BAT_ENV_ENV
                             else if (BAT_ENV_PROP != null) BAT_ENV_PROP
                             else 'anypoint.mulesoft.com'

fun hasMetadata(payload): Boolean = payload.metadata != null
                                    and payload.metadata != ''
                                    and payload.metadata.testId != null
                                    and payload.metadata.testId != ''
                                    and payload.metadata.executionId != null
                                    and payload.metadata.executionId != ''

