package com.mulesoft.cloudhub.client;

import java.util.List;

import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MediaType;

import org.apache.commons.lang3.StringUtils;
import org.codehaus.jackson.jaxrs.JacksonJsonProvider;
import org.codehaus.jackson.map.DeserializationConfig;
import org.codehaus.jackson.map.annotate.JsonSerialize.Inclusion;

import com.google.gson.Gson;
import com.mulesoft.ch.rest.model.Account;
import com.mulesoft.ch.rest.model.Application;
import com.mulesoft.ch.rest.model.DomainStatus;
import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.GenericType;
import com.sun.jersey.api.client.WebResource;
import com.sun.jersey.api.client.WebResource.Builder;
import com.sun.jersey.api.client.config.ClientConfig;
import com.sun.jersey.api.client.config.DefaultClientConfig;
import com.sun.jersey.api.client.filter.LoggingFilter;
import com.sun.jersey.multipart.impl.MultiPartWriter;

public class CloudHubConnectionImpl implements CloudHubConnectionI {


    public static final String NEW_DEFAULT_URL = "https://anypoint.mulesoft.com/cloudhub/";
    public static final String ENVIRONMENT_ID = "X-ANYPNT-ENV-ID";

    protected String url;
    protected Client client;

    protected String username;
    protected String password;
    protected String apiToken;
    protected String environmentId;

    protected String accessToken;
    protected Gson gson;

    protected void init(String url, boolean debug) {
        if (StringUtils.isBlank(url)) {

            this.url = NEW_DEFAULT_URL;
        } else {
            this.url = url.endsWith("/") ? url : url + "/";
        }

        this.client = Client.create(getClientConfig());

        if (debug) {
            this.client.addFilter(new LoggingFilter());
        }
        gson = new Gson();
    }

    protected CloudHubConnectionImpl() {
    }

    public CloudHubConnectionImpl(String url, String accessToken,String environmentId, boolean debug) {
        this.environmentId = environmentId;
        this.accessToken = accessToken;
        this.init(url, debug);
    }

    public CloudHubConnectionImpl(String url, String accessToken, boolean debug) {
        this(url,accessToken,"",debug);
    }

    @Override
    public CloudHubDomainConnectionI connectWithDomain(String domain) {

        if (!StringUtils.isBlank(domain)) {
            return new CloudHubDomainConnectionImpl(this,environmentId, true, domain);

        } else {
            throw new IllegalArgumentException("Domain must not be null or empty");
        }

    }

    @Override
    public Account retrieveAccount() throws CloudHubException {
        ClientResponse clientResponse = createBuilder("account/").type(MediaType.APPLICATION_JSON_TYPE).get(ClientResponse.class);
        if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus())) {
            return clientResponse.getEntity(Account.class);
        }

        throw buildExpception(clientResponse);
    }

    @Override
    public boolean isDomainAvailable(String domain) throws CloudHubException {
        ClientResponse clientResponse = addEnvironmentHeader(createBuilder("applications/domains/" + domain)).get(ClientResponse.class);
        if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus())) {
            return clientResponse.getEntity(DomainStatus.class).isAvailable();
        }

        throw buildExpception(clientResponse);
    }

    @Override
    public List<Application> retrieveApplications() {
        ClientResponse clientResponse = addEnvironmentHeader(createApplicationBuilder("")).get(ClientResponse.class);
        if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus())) {
            return clientResponse.getEntity(new GenericType<List<Application>>(){});
        }
        throw buildExpception(clientResponse);
    }

    @Override
    public Application createApplication(Application application) throws CloudHubException {

        if (isDomainAvailable(application.getDomain())) {

            ClientResponse clientResponse = addEnvironmentHeader(createApplicationBuilder("").type(MediaType.APPLICATION_JSON_TYPE)).post(ClientResponse.class, application);

            if (ClientResponse.Status.CREATED.equals(clientResponse.getClientResponseStatus())) {
                return clientResponse.getEntity(Application.class);
            }

            throw buildExpception(clientResponse);
        }

        throw new CloudHubException("The application's domain is not available", String.valueOf(ClientResponse.Status.CONFLICT.getStatusCode()));

    }

    @SuppressWarnings("unchecked")
    @Override
    public List<String> getSupportedMuleVersions() throws CloudHubException {
        ClientResponse clientResponse = addEnvironmentHeader(createApplicationBuilder("supportedMuleVersions")).get(ClientResponse.class);
        if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus())) {
            return clientResponse.getEntity(List.class);
        }

        throw buildExpception(clientResponse);
    }

    private ClientConfig getClientConfig() {
        final ClientConfig clientConfig = new DefaultClientConfig();
        clientConfig.getClasses().add(JacksonJsonProvider.class);
        ObjectMapper mapper = new ObjectMapper();
        mapper.configure(DeserializationConfig.Feature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        JacksonJsonProvider jsonProvider = new JacksonJsonProvider(mapper);
        clientConfig.getSingletons().add(jsonProvider);
        clientConfig.getClasses().add(MultiPartWriter.class);

        return clientConfig;
    }

    protected Builder createBuilder(final String path) {
        return authorizeResource(createResource(path));
    }

    protected Builder authorizeResource(WebResource pathResource) {
        return pathResource.header(HttpHeaders.AUTHORIZATION, "Bearer " + accessToken);
    }

    protected Builder addEnvironmentHeader(Builder pathResource) {
        return pathResource.header(ENVIRONMENT_ID, environmentId);
    }


    protected WebResource createResource(final String path) {
        return this.client.resource(this.url + "api/").path(path);
    }

    protected final WebResource.Builder createApplicationBuilder(final String path) {
        return createBuilder("applications/" + path);
    }

    public static class ObjectMapper extends org.codehaus.jackson.map.ObjectMapper {

        public ObjectMapper() {
            super();
            getSerializationConfig().setSerializationInclusion(Inclusion.NON_NULL);
        }

    }

    protected CloudHubException buildExpception(ClientResponse clientResponse) throws CloudHubException {
        String message = "";
        int status = clientResponse.getStatus();

        if (ClientResponse.Status.BAD_REQUEST.equals(clientResponse.getClientResponseStatus())) {
            message = gson.fromJson(clientResponse.getEntity(String.class), ReturnMessage.class).getErrorMessage();
        } else {
            message = clientResponse.getClientResponseStatus().getReasonPhrase();
        }
        return new CloudHubException(message, String.valueOf(status));

    }

    @Override
    public String getUrl() {
        return this.url;
    }

    @Override
    public String getUsername() {
        return this.username;
    }

    @Override
    public String getPassowrd() {
        return this.password;
    }

    @Override
    public String getAccessToken() {
        return this.accessToken;
    }


}
