package com.mulesoft.cloudhub.client;

import java.io.File;
import java.io.IOException;

import javax.ws.rs.core.MediaType;

import com.mulesoft.ch.rest.model.Application;
import com.mulesoft.ch.rest.model.ApplicationStatus;
import com.mulesoft.ch.rest.model.ApplicationStatusChange;
import com.mulesoft.ch.rest.model.ApplicationUpdateInfo;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.WebResource;
import com.sun.jersey.multipart.FormDataMultiPart;
import com.sun.jersey.multipart.file.FileDataBodyPart;

public class CloudHubDomainConnectionImpl extends CloudHubConnectionImpl implements CloudHubDomainConnectionI {

    private String domain;
    private String environmentId;


    protected CloudHubDomainConnectionImpl(CloudHubConnectionI con,String environmentId, boolean csAuthentication, String domain) {
        super(con.getUrl(), con.getAccessToken(),environmentId, false);
        this.domain = domain;
        this.environmentId = environmentId;
    }

    public CloudHubDomainConnectionImpl(String url, String accessToken, boolean debug) {
        super();
        this.accessToken = accessToken;
        this.init(url, debug);

    }

    @Override
    public String getDomain() {
        return domain;
    }

    @Override
    public Application retrieveApplication() throws CloudHubException {
		ClientResponse clientResponse = null;
        try {
        	clientResponse = createApplicationDomainBuilder("").get(ClientResponse.class);

        	if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus()))
                return clientResponse.getEntity(Application.class);

            throw buildExpception(clientResponse);
            
        } finally {
        	closeSilently(clientResponse);
        	clientResponse = null;
        }
    }

    @Override
    public ApplicationStatus retrieveApplicationStatus() throws CloudHubException {
		ClientResponse clientResponse = null;
        try {
        	clientResponse = createApplicationDomainBuilder("status").get(ClientResponse.class);

            if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus()))
                return ApplicationStatus.valueOf(clientResponse.getEntity(String.class));

            throw buildExpception(clientResponse);
        } finally {
        	closeSilently(clientResponse);
        	clientResponse = null;
        }
    }


    @Override
    public void deployApplication(File file, long maxWaitTime) {
        FormDataMultiPart multiPart = null;
        ClientResponse clientResponse = null;
        try {
            multiPart = new FormDataMultiPart();

            multiPart.bodyPart(new FileDataBodyPart("file", file, MediaType.APPLICATION_OCTET_STREAM_TYPE));
            clientResponse = createApplicationDomainBuilderV2("files").type(MediaType.MULTIPART_FORM_DATA).post(ClientResponse.class, multiPart);

            if (!ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus()))
                throw buildExpception(clientResponse);
            
            waitForFinish(maxWaitTime, failedBefore());
        } finally {
        	closeSilently(multiPart);
        	closeSilently(clientResponse);
        	multiPart = null;
        	clientResponse = null;
        }
    }

	private void closeSilently(ClientResponse clientResponse) {
		if (clientResponse != null) {
			try {
				clientResponse.close();
			} finally {
				clientResponse = null;
			}
		}
	}

	private void closeSilently(FormDataMultiPart multiPart) {
		if (multiPart != null) {
			try {
				multiPart.close();
			} catch(IOException ioEx) {
				multiPart.cleanup(); // Second try!
			} finally {
				multiPart = null;
			}
		}
	}
    
    @Override
    public Application updateApplication(ApplicationUpdateInfo newApplicationInfo) throws CloudHubException {

		ClientResponse clientResponse = null;
        try {
            Application application = retrieveApplication();
            ApplicationUpdateInfo oldApplicationInfo = new ApplicationUpdateInfo(application);

            if (!oldApplicationInfo.equals(newApplicationInfo)) {
                clientResponse = createApplicationDomainBuilder("").type(MediaType.APPLICATION_JSON_TYPE).put(ClientResponse.class, newApplicationInfo);

                if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus()))
                    return clientResponse.getEntity(Application.class);

                throw buildExpception(clientResponse);
            }

            return application;
        } finally {
        	closeSilently(clientResponse);
        	clientResponse = null;
        }
    }

    @Override
    public void updateApplicationStatus(ApplicationStatusChange applicationStatusChange, long maxWaitTime) throws CloudHubException {
		ClientResponse clientResponse = null;
        try {
        	boolean failedBefore = failedBefore();
            clientResponse = createApplicationDomainBuilder("status").type(MediaType.APPLICATION_JSON_TYPE).post(ClientResponse.class, applicationStatusChange);

            if (!ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus())) {
                throw buildExpception(clientResponse);
            }

            waitForFinish(maxWaitTime, failedBefore);
        } finally {
        	closeSilently(clientResponse);
        	clientResponse = null;
        }
    }

    @Override
    public void deleteApplication() throws CloudHubException {
		ClientResponse clientResponse = null;
        try {
            clientResponse = createApplicationDomainBuilder("").delete(ClientResponse.class);

            if (!ClientResponse.Status.NO_CONTENT.equals(clientResponse.getClientResponseStatus())) {
                throw buildExpception(clientResponse);
            }
        } finally {
        	closeSilently(clientResponse);
        	clientResponse = null;
        }
    }

    private final WebResource.Builder createApplicationDomainBuilder(final String path) {
        return addEnvironmentHeader(createBuilder("applications/" + domain + "/" + path));
    }

    private final WebResource.Builder createApplicationDomainBuilderV2(final String path) {
        return addEnvironmentHeader(createBuilder("v2/applications/" + domain + "/" + path));
    }
    

    private void waitForFinish(long maxWaitTime, boolean failedBefore) {
        if (maxWaitTime == 0)
            return;
        else if (maxWaitTime == 1) { // WTF? Why 1 is turned into a very long wait time????
            maxWaitTime = Long.MAX_VALUE;
        }

        final long before = System.currentTimeMillis();
        while (System.currentTimeMillis() - before < maxWaitTime) {
            final Application latestApplication = retrieveApplication();
            final ApplicationStatus status = latestApplication.getDeploymentUpdateStatus();

            if (!failedBefore && status != null) {
                if (status.equals(ApplicationStatus.DEPLOY_FAILED)) {
                    throw new CloudHubException("Application " + domain + " failed to start.", String.valueOf(ClientResponse.Status.EXPECTATION_FAILED.getStatusCode()));
                }
            }
            try {
                Thread.sleep(1000L);
            } catch (InterruptedException e) {
                // Quit loop when interrupted
                break;
            }
        }

        final Application latestApplication = retrieveApplication();
        final ApplicationStatus status = latestApplication.getDeploymentUpdateStatus();

        if (status == null && latestApplication.getStatus().equals(ApplicationStatus.STARTED))
            return;
        if (status == null && latestApplication.getStatus().equals(ApplicationStatus.UNDEPLOYED))
            return;

        if (status.equals(ApplicationStatus.DEPLOY_FAILED)) {
            throw new CloudHubException("Application " + domain + " failed to start.", String.valueOf(ClientResponse.Status.EXPECTATION_FAILED.getStatusCode()));
        }

        throw new CloudHubException("CloudHub did not finish the process before the specified time", String.valueOf(ClientResponse.Status.GATEWAY_TIMEOUT.getStatusCode()));
    }

    private boolean failedBefore() {
        ApplicationStatus status = retrieveApplication().getDeploymentUpdateStatus();
        return status == null ? false : status.equals(ApplicationStatus.DEPLOY_FAILED);
    }

}
