
package com.mulesoft.cloudhub.client;

import java.io.File;

import com.mulesoft.ch.rest.model.Application;
import com.mulesoft.ch.rest.model.ApplicationStatus;
import com.mulesoft.ch.rest.model.ApplicationStatusChange;
import com.mulesoft.ch.rest.model.ApplicationUpdateInfo;
import com.mulesoft.cloudhub.client.annotation.Component;
import com.mulesoft.cloudhub.client.annotation.UsedBy;

public interface CloudHubDomainConnectionI extends CloudHubConnectionI{
	
	/**
	 * Returns the connected domain
	 * 
	 * @return the connected domain
	 */
	public String getDomain();
	
	/**
	 * Retrieves the application based on the connected domain
	 * 
	 * @return an instance of {@link com.mulesoft.ch.rest.model.Application}
	 *  
	 * @throws CloudHubException if the domain does not exist
	 */
	@UsedBy(Component.STUDIO)
	public Application retrieveApplication() throws CloudHubException;
	
	/**
	 * Retrieves the application's status
	 * 
	 * @return one of {@link com.mulesoft.ch.rest.model.Application.ApplicationStatus}
	 * @throws CloudHubException if there was an error when retrieving the application status
	 */
	public ApplicationStatus retrieveApplicationStatus() throws CloudHubException;
	

	/**
	 * Deploys an application with the provided fileStream
	 * 
	 * @param file the file to be deployed
	 * @param waitMaxTime Maximum time allowed to deploy/undeploy in miliseconds. 
	 * Case 1: 0, the connector won't wait CH to finish the deploy/undeploy process and will return a response based on CH API response status.
	 * Case 2: 1, the connector will wait until CH finish the deploy/undeploy process and will throw an exception if the deploy fails.
	 * Case 3: anyOthervalue, the connector will wait that amount of miliseconds for CH to finish the deploy/undeploy process.
	 * If takes longer, it will throw an exception. If takes shorter, it will follow the same process as case 2. 
	 * 
	 * @throws CloudHubException if there was an error while deploying the application
	 */
	@UsedBy(value = {Component.STUDIO,Component.DEPLOY_MAVEN_PLUGIN})
	public void deployApplication(File file, long waitMaxTime) throws CloudHubException;
	
	/**
	 * Updates an application's information on your CloudHub account.
	 * This method just updates metaData like Mule Version, Description, etc. 
	 * If you want to update a new stream, use {@link #deployApplication} instead
	 * 
	 * @param applicationInfo an instance of {@link com.mulesoft.ch.rest.model.ApplicationUpdateInfo} with the information to be updated
	 * 
	 * @return an instance of {@link com.mulesoft.ch.rest.model.Application} containing all the new application's information provided by CloudHub
	 * 
	 * @throws CloudHubException if there was an error while updating the application
	 */
	@UsedBy(value = {Component.STUDIO,Component.DEPLOY_MAVEN_PLUGIN})
	public Application updateApplication(ApplicationUpdateInfo application) throws CloudHubException;
	
	/**
	 * Updates an application's status based on some specific commands. Can be START or STOP
	 * 
	 * @param applicationStatus an instance of {@link com.mulesoft.cloudhub.client.ApplicationStatusChange} with the new application status
	 * @param waitMaxTime Maximum time allowed to deploy/undeploy in miliseconds. 
	 * Case 1: 0, the connector won't wait CH to finish the deploy/undeploy process and will return a response based on CH API response status.
	 * Case 2: 1, the connector will wait until CH finish the deploy/undeploy process and will throw an exception if the deploy fails.
	 * Case 3: anyOthervalue, the connector will wait that amount of miliseconds for CH to finish the deploy/undeploy process.
	 * If takes longer, it will throw an exception. If takes shorter, it will follow the same process as case 2. 
	 * 
	 * @throws CloudHubException if there was an error while updating the application's status.
	 */
	@UsedBy(Component.DEPLOY_MAVEN_PLUGIN)
	public void updateApplicationStatus(ApplicationStatusChange applicationStatus, long waitMaxTime) throws CloudHubException;
	
	/**
	 * Deletes the application by using the connection's domain.
	 * The domain can not be invalid since it was validated when creating the connection.
	 * 
	 * @throws CloudHubException if there was an error while deleting the application
	 */
	public void deleteApplication() throws CloudHubException;
	

}
