package com.mulesoft.cloudhub.client;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.util.List;
import java.util.Map;

import javax.ws.rs.core.MediaType;

import com.mulesoft.ch.rest.model.Application;
import com.mulesoft.ch.rest.model.ApplicationStatus;
import com.mulesoft.ch.rest.model.ApplicationStatusChange;
import com.mulesoft.ch.rest.model.ApplicationUpdateInfo;
import com.mulesoft.ch.rest.model.LogResults;
import com.mulesoft.ch.rest.model.Tenant;
import com.mulesoft.ch.rest.model.TenantResults;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.WebResource;

public class CloudHubDomainConnectionImpl extends CloudHubConnectionImpl implements CloudHubDomainConnectionI {
	
    private String domain;

    protected CloudHubDomainConnectionImpl(CloudHubConnectionI con, String domain) {
        super(con.getUrl(), con.getUsername(), con.getPassword(),
              null, con.getReadTimeout(), con.getConnectionTimeout(), false);
        this.domain = domain;
    }
    
    protected CloudHubDomainConnectionImpl(CloudHubConnectionI con, boolean csAuthentication, String domain) {
        super(con.getUrl(), con.getAccessToken(), con.getEnvironmentId(), false);
        this.domain = domain;
    }

    public CloudHubDomainConnectionImpl(String url, String accessToken, boolean debug) {
        super();
        this.accessToken = accessToken;
        this.init(url, debug);
    }

    public CloudHubDomainConnectionImpl(String url, String accessToken, String environmentId, boolean debug) {
        super();
        this.accessToken = accessToken;
        this.environmentId = environmentId;
        this.init(url, debug);
    }
    
    @Override
    public String getDomain() {
        return domain;
    }

    @Override
    public Application retrieveApplication() throws CloudHubException {
        ClientResponse clientResponse = createApplicationDomainBuilder("").get(ClientResponse.class);

        if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus()))
            return clientResponse.getEntity(Application.class);

        throw buildExpception(clientResponse);

    }

    @Override
    public LogResults retrieveApplicationLog(Map<String, String> queryParams) throws CloudHubException {

        WebResource resource = createResource("/applications/" + domain + "/logs");

        if (queryParams != null && !queryParams.isEmpty()) {
        	for (String key : queryParams.keySet()) {
        		resource = resource.queryParam(key, queryParams.get(key));
        	}
        }
        
        ClientResponse clientResponse = authorizeResource(resource).get(ClientResponse.class);

        if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus()))
            return clientResponse.getEntity(LogResults.class);

        throw buildExpception(clientResponse);

    }
    
    @Override
    public File downloadApplicationLog(Map<String, String> queryParams) throws CloudHubException {

        WebResource resource = createResource("/applications/" + domain + "/logs/download");

        if (queryParams != null && !queryParams.isEmpty()) {
        	for (String key : queryParams.keySet()) {
        		resource = resource.queryParam(key, queryParams.get(key));
        	}
        }
        
        ClientResponse clientResponse = authorizeResource(resource).get(ClientResponse.class);

        if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus()))
            return clientResponse.getEntity(File.class);

        throw buildExpception(clientResponse);

    }
    
    @Override
    public ApplicationStatus retrieveApplicationStatus() throws CloudHubException {
        ClientResponse clientResponse = createApplicationDomainBuilder("status").get(ClientResponse.class);

        if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus()))
            return ApplicationStatus.valueOf(clientResponse.getEntity(String.class));

        throw buildExpception(clientResponse);
    }

    @Override
    public void deployApplication(InputStream fileStream, long maxWaitTime) {
        boolean failedBefore = failedBefore();

        ClientResponse clientResponse = createApplicationDomainBuilder("deploy").type(MediaType.APPLICATION_OCTET_STREAM_TYPE).post(ClientResponse.class, fileStream);

        if (!ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus()))
            throw buildExpception(clientResponse);


        waitForFinish(maxWaitTime, failedBefore);
    }

    @Override
    public void deployApplication(File file, long maxWaitTime) {
        try {
            this.deployApplication(new FileInputStream(file), maxWaitTime);
        } catch (FileNotFoundException e) {
            throw new RuntimeException("File does not exist", e);
        }
    }

    @Override
    public Application updateApplication(ApplicationUpdateInfo newApplicationInfo) throws CloudHubException {

        Application application = retrieveApplication();
        ApplicationUpdateInfo oldApplicationInfo = new ApplicationUpdateInfo(application);

        if (!oldApplicationInfo.equals(newApplicationInfo)) {
            ClientResponse clientResponse = createApplicationDomainBuilder("").type(MediaType.APPLICATION_JSON_TYPE).put(ClientResponse.class, newApplicationInfo);

            if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus()))
                return clientResponse.getEntity(Application.class);

            throw buildExpception(clientResponse);
        }

        return application;

    }

    @Override
    public void updateApplicationStatus(ApplicationStatusChange applicationStatusChange, long maxWaitTime) throws CloudHubException {
        boolean failedBefore = failedBefore();
        ClientResponse clientResponse = createApplicationDomainBuilder("status").type(MediaType.APPLICATION_JSON_TYPE).post(ClientResponse.class, applicationStatusChange);

        if (!ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus())) {
            throw buildExpception(clientResponse);
        }

        waitForFinish(maxWaitTime, failedBefore);
    }

    @Override
    public void deleteApplication() throws CloudHubException {
        ClientResponse clientResponse = createApplicationDomainBuilder().delete(ClientResponse.class);

        if (!ClientResponse.Status.NO_CONTENT.equals(clientResponse.getClientResponseStatus())) {
            throw buildExpception(clientResponse);
        }

    }

    @Override
    public TenantResults retrieveTenants(Integer limit, Integer offset,
                                         String query, Boolean enabled) throws CloudHubException {

        WebResource resource = createResource("/applications/" + domain + "/tenants");

        if (limit != null) {
            resource = resource.queryParam("limit", String.valueOf(limit));
        }

        if (offset != null) {
            resource = resource.queryParam("offset", String.valueOf(offset));
        }

        if (enabled != null) {
            resource = resource.queryParam("enabled", enabled.toString());
        }

        if (query != null) {
            resource = resource.queryParam("search", query);
        }

        ClientResponse clientResponse = authorizeResource(resource).type(MediaType.APPLICATION_JSON_TYPE).get(ClientResponse.class);

        if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus()))
            return clientResponse.getEntity(TenantResults.class);

        throw buildExpception(clientResponse);

    }

    @Override
    public Tenant retrieveTenant(String tenantId) throws CloudHubException {
        ClientResponse clientResponse = createApplicationDomainTenantBuilder(tenantId).type(MediaType.APPLICATION_JSON_TYPE).get(ClientResponse.class);

        if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus()))
            return clientResponse.getEntity(Tenant.class);

        throw buildExpception(clientResponse);

    }

    @Override
    public InputStream retrieveTenantClientCertificates(String tenantId) throws CloudHubException {
        ClientResponse clientResponse = createApplicationDomainTenantBuilder(tenantId + "/openVPNConfigs/").type(MediaType.APPLICATION_JSON_TYPE).get(ClientResponse.class);
        return clientResponse.getEntity(InputStream.class);
    }

    @Override
    public Tenant createTenant(Tenant tenant) throws CloudHubException {
        ClientResponse clientResponse = createApplicationDomainTenantBuilder("").type(MediaType.APPLICATION_JSON_TYPE).post(ClientResponse.class, tenant);

        if (ClientResponse.Status.CREATED.equals(clientResponse.getClientResponseStatus()))
            return clientResponse.getEntity(Tenant.class);

        throw buildExpception(clientResponse);

    }

    @Override
    public Tenant updateTenant(Tenant tenant) throws CloudHubException {
        ClientResponse clientResponse = createApplicationDomainTenantBuilder(tenant.getId()).type(MediaType.APPLICATION_JSON_TYPE).put(ClientResponse.class, tenant);

        if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus()))
            return clientResponse.getEntity(Tenant.class);

        throw buildExpception(clientResponse);
    }

    @Override
    public void deleteTenant(String tenantId) throws CloudHubException {
        ClientResponse clientResponse = createApplicationDomainTenantBuilder(tenantId).type(MediaType.APPLICATION_JSON_TYPE).delete(ClientResponse.class);
        if (!ClientResponse.Status.NO_CONTENT.equals(clientResponse.getClientResponseStatus())) {
            throw buildExpception(clientResponse);
        }
    }

    @Override
    public void deleteTenants(List<String> tenantIds) throws CloudHubException {
        ClientResponse clientResponse = createApplicationDomainTenantBuilder("").type(MediaType.APPLICATION_JSON_TYPE).delete(ClientResponse.class, tenantIds);
        if (!ClientResponse.Status.NO_CONTENT.equals(clientResponse.getClientResponseStatus())) {
            throw buildExpception(clientResponse);
        }

    }

    @Override
    public void deleteTenantClientCertificates(String tenantId) throws CloudHubException {
        ClientResponse clientResponse = createApplicationDomainTenantBuilder(tenantId + "/openVPNConfigs/").type(MediaType.APPLICATION_JSON_TYPE).delete(ClientResponse.class);
        if (!ClientResponse.Status.NO_CONTENT.equals(clientResponse.getClientResponseStatus())) {
            throw buildExpception(clientResponse);
        }

    }

    private final WebResource.Builder createApplicationDomainBuilder(final String path) {
        return createBuilder("applications/" + domain + "/" + path);
    }

    private final WebResource.Builder createApplicationDomainBuilder() {
        return createBuilder("applications/" + domain);
    }
    
    private final WebResource.Builder createApplicationDomainTenantBuilder(final String path) {
        return createBuilder("applications/" + domain + "/tenants/" + path);
    }

    private void waitForFinish(long maxWaitTime, boolean failedBefore) {
        if (maxWaitTime == 0)
            return;
        else if (maxWaitTime == 1) {
            maxWaitTime = Long.MAX_VALUE;
        }

        final long before = System.currentTimeMillis();
        while (System.currentTimeMillis() - before < maxWaitTime) {
            final Application latestApplication = retrieveApplication();
            final ApplicationStatus status = latestApplication.getDeploymentUpdateStatus();

            if (!failedBefore && status != null) {
                if (status.equals(ApplicationStatus.DEPLOY_FAILED)) {
                    throw new CloudHubException("Application " + domain + " failed to start.", String.valueOf(ClientResponse.Status.EXPECTATION_FAILED.getStatusCode()));
                }
            }
            try {
                Thread.sleep(1000L);
            } catch (InterruptedException e) {
                //Quit loop when interrupted
                break;
            }
        }


        final Application latestApplication = retrieveApplication();
        final ApplicationStatus status = latestApplication.getDeploymentUpdateStatus();

        if (status == null && latestApplication.getStatus().equals(ApplicationStatus.STARTED)) return;
        if (status == null && latestApplication.getStatus().equals(ApplicationStatus.UNDEPLOYED)) return;

        if (status.equals(ApplicationStatus.DEPLOY_FAILED)) {
            throw new CloudHubException("Application " + domain + " failed to start.", String.valueOf(ClientResponse.Status.EXPECTATION_FAILED.getStatusCode()));
        }

        throw new CloudHubException("CloudHub did not finish the process before the specified time", String.valueOf(ClientResponse.Status.GATEWAY_TIMEOUT.getStatusCode()));
    }

    private boolean failedBefore() {
        ApplicationStatus status = retrieveApplication().getDeploymentUpdateStatus();
        return status == null ? false : status.equals(ApplicationStatus.DEPLOY_FAILED);
    }

}
