package com.mulesoft.cloudhub.client;

import com.mulesoft.ch.rest.model.*;
import com.mulesoft.ch.rest.model.Notification.NotificationStatus;
import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.WebResource;
import com.sun.jersey.api.client.WebResource.Builder;
import com.sun.jersey.api.client.config.ClientConfig;
import com.sun.jersey.api.client.config.DefaultClientConfig;
import com.sun.jersey.api.client.filter.LoggingFilter;
import com.sun.jersey.api.json.JSONConfiguration;
import com.sun.jersey.core.util.Base64;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.codehaus.jackson.jaxrs.JacksonJsonProvider;
import org.codehaus.jackson.map.DeserializationConfig;
import org.codehaus.jackson.map.annotate.JsonSerialize.Inclusion;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MediaType;

import java.net.URI;
import java.nio.charset.Charset;
import java.util.Collection;
import java.util.List;

public class CloudHubConnectionImpl implements CloudHubConnectionI {

    protected Logger logger = LoggerFactory.getLogger(CloudHubConnectionImpl.class);

    @Deprecated
    public static final String DEFAULT_URL = "https://cloudhub.io/";
    public static final String NEW_DEFAULT_URL = "https://anypoint.mulesoft.com/cloudhub/";

    protected String url;
    protected Client client;

    protected String username;
    protected String password;
    protected String apiToken;

    protected String accessToken;
    protected String environmentId;
    protected boolean csAuthentication = false;


    protected void init(String url, boolean debug) {
        if (StringUtils.isBlank(url)) {
            if (csAuthentication) {
                this.url = NEW_DEFAULT_URL;
            } else {
                this.url = DEFAULT_URL;
            }
        } else {
            this.url = url.endsWith("/") ? url : url + "/";
        }

        this.client = Client.create(getClientConfig());

        if (debug) {
            this.client.addFilter(new LoggingFilter());
        }
    }

    protected CloudHubConnectionImpl() {
    }

    public CloudHubConnectionImpl(String url, String accessToken, boolean debug) {
        this.csAuthentication = true;
        this.accessToken = accessToken;
        this.init(url, debug);
    }

    public CloudHubConnectionImpl(String url, String accessToken, String environmentId, boolean debug) {
        this.csAuthentication = true;
        this.accessToken = accessToken;
        this.environmentId = environmentId;
        this.init(url, debug);
    }
    
    public CloudHubConnectionImpl(String url, String username, String password, String sandbox, boolean debug) {

        if (StringUtils.isBlank(username)) {
            apiToken = System.getProperty("cloudhub.api.token") != null ? System.getProperty("cloudhub.api.token") : System.getProperty("ion.api.token");

            Validate.notNull(apiToken, "apiToken can not be null if username was not provided");
            logger.debug("Using CloudHub token authentication.");
        } else {
            Validate.notNull(password, "Password can not be null if username is provided.");
            logger.debug("Using CloudHub username/password authentication because the username is set.");
        }

        this.username = StringUtils.isNotBlank(sandbox) ? username + "@" + sandbox : username;
        this.password = password;

        this.init(url, debug);
    }

    //TODO: validate if we can remove this method
    @Override
    public CloudHubConnectionI createConnection(String url, String username,
                                                String password, String sandbox) {
        return new CloudHubConnectionImpl(url, username, password, sandbox, false);
    }

    @Override
    public CloudHubDomainConnectionI connectWithDomain(String domain) {

        if (!StringUtils.isBlank(domain)) {
            if(csAuthentication){
                return new CloudHubDomainConnectionImpl(this, true, domain);
            }   else{
                return new CloudHubDomainConnectionImpl(this, domain);
            }
        } else {
            throw new IllegalArgumentException("Domain must not be null or empty");
        }

    }

    @Override
    public Account retrieveAccount() throws CloudHubException {
        ClientResponse clientResponse = createBuilder("account/").type(MediaType.APPLICATION_JSON_TYPE).get(ClientResponse.class);
        if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus())) {
            return clientResponse.getEntity(Account.class);
        }

        throw buildException(clientResponse);
    }

    @Override
    public boolean isDomainAvailable(String domain) throws CloudHubException {
        ClientResponse clientResponse = createBuilder("applications/domains/" + domain).get(ClientResponse.class);
        if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus())) {
            return clientResponse.getEntity(DomainStatus.class).isAvailable();
        }

        throw buildException(clientResponse);
    }

    @SuppressWarnings("unchecked")
    @Override
    public Collection<Application> retrieveApplications() {
        ClientResponse clientResponse = createApplicationBuilder("").get(ClientResponse.class);
        if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus())) {
            return clientResponse.getEntity(Collection.class);
        }

        throw buildException(clientResponse);
    }

    @Override
    public Application createApplication(Application application)
            throws CloudHubException {

        if (isDomainAvailable(application.getDomain())) {

            ClientResponse clientResponse = createApplicationBuilder("").type(MediaType.APPLICATION_JSON_TYPE).post(ClientResponse.class, application);

            if (ClientResponse.Status.CREATED.equals(clientResponse.getClientResponseStatus())) {
                return clientResponse.getEntity(Application.class);
            }

            throw buildException(clientResponse);
        }

        throw new CloudHubException("The application's domain is not available", String.valueOf(ClientResponse.Status.CONFLICT.getStatusCode()));

    }

    @Override
    public NotificationResults retrieveNotifications(String domain, String tenantId, Integer limit,
                                                     Integer offset, NotificationStatus status, String message)
            throws CloudHubException {

        Validate.notNull(domain, "The domain can not be null");

        WebResource resource = createResource("/notifications");

        resource = resource.queryParam("domain", domain);

        if (tenantId != null) {
            resource = resource.queryParam("tenantId", String.valueOf(tenantId));
        }

        if (limit != null) {
            resource = resource.queryParam("limit", String.valueOf(limit));
        }

        if (offset != null) {
            resource = resource.queryParam("offset", String.valueOf(offset));
        }

        if (status != null) {
            resource = resource.queryParam("status", status.toString());
        }

        if (message != null) {
            resource = resource.queryParam("search", message);
        }

        ClientResponse clientResponse = authorizeResource(resource).type(MediaType.APPLICATION_JSON_TYPE).get(ClientResponse.class);

        if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus())) {
            return clientResponse.getEntity(NotificationResults.class);
        }

        throw buildException(clientResponse);
    }

    @Override
    public Notification retrieveNotification(String notificationId)
            throws CloudHubException {
        ClientResponse clientResponse = createNotificationBuilder(notificationId).type(MediaType.APPLICATION_JSON_TYPE).get(ClientResponse.class);

        if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus())) {
            return clientResponse.getEntity(Notification.class);
        }

        throw buildException(clientResponse);


    }

    @Override
    public Notification createNotification(Notification notification)
            throws CloudHubException {

        ClientResponse clientResponse = createNotificationBuilder("").type(MediaType.APPLICATION_JSON_TYPE).post(ClientResponse.class, notification);

        if (ClientResponse.Status.CREATED.equals(clientResponse.getClientResponseStatus())) {
            URI uri = clientResponse.getLocation();
            return new Notification(uri.getPath().substring(uri.getPath().lastIndexOf("/") + 1, uri.getPath().length()));
        }

        throw buildException(clientResponse);
    }

    @Override
    public void updateNotificationsStatus(String domain, String tenantId,
                                          NotificationStatus status) throws CloudHubException {

        Validate.notNull(domain, "The domain can not be null");
        Validate.notNull(status, "The status can not be null");

        WebResource resource = createResource("notifications/").queryParam("domain", domain);

        if (tenantId != null) {
            resource.queryParam("tenantId", tenantId);
        }

        ClientResponse clientResponse = authorizeResource(resource).type(MediaType.APPLICATION_JSON_TYPE).put(ClientResponse.class, status);

        if (!ClientResponse.Status.NO_CONTENT.equals(clientResponse.getClientResponseStatus())) {
            throw buildException(clientResponse);
        }
    }

    @Override
    public void updateNotificationStatus(String notificationId,
                                         NotificationStatus status) throws CloudHubException {

        ClientResponse clientResponse = createNotificationBuilder(notificationId).type(MediaType.APPLICATION_JSON_TYPE).put(ClientResponse.class, status);
        if (!ClientResponse.Status.NO_CONTENT.equals(clientResponse.getClientResponseStatus())) {
            buildException(clientResponse);
        }
    }

    @SuppressWarnings("unchecked")
    @Override
    public List<String> getSupportedMuleVersions() throws CloudHubException {
        ClientResponse clientResponse = createApplicationBuilder("supportedMuleVersions").get(ClientResponse.class);
        if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus())) {
            return clientResponse.getEntity(List.class);
        }

        throw buildException(clientResponse);
    }

    private ClientConfig getClientConfig() {
        //Ensure we have all required parameters
        final ClientConfig clientConfig = new DefaultClientConfig();
        clientConfig.getFeatures().put(JSONConfiguration.FEATURE_POJO_MAPPING,Boolean.TRUE);
        clientConfig.getClasses().add(JacksonJsonProvider.class);
        ObjectMapper mapper = new ObjectMapper();
        mapper.configure(DeserializationConfig.Feature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        JacksonJsonProvider jsonProvider = new JacksonJsonProvider(mapper);
        clientConfig.getSingletons().add(jsonProvider);

        return clientConfig;
    }

    protected Builder createBuilder(final String path) {
        return authorizeResource(createResource(path));
    }

    protected Builder authorizeResource(WebResource pathResource) {
        
    	//enable for logging
    	//pathResource.addFilter(new LoggingFilter());
    	
    	if (csAuthentication) {
    		if (environmentId != null) {
                return pathResource.header(HttpHeaders.AUTHORIZATION, "Bearer " + accessToken).header("X-ANYPNT-ENV-ID", environmentId);
            } else {
            	return pathResource.header(HttpHeaders.AUTHORIZATION, "Bearer " + accessToken);
            }
        }
    	
        if (apiToken == null) {
            return pathResource.header(HttpHeaders.AUTHORIZATION, "Basic " + new String(Base64.encode(this.username + ":" + this.password), Charset.forName("ASCII")));
        } else {
            return pathResource.header("X-ION-Authenticate", apiToken);
        }
    }

    protected WebResource createResource(final String path) {
        return this.client.resource(this.url + "api/").path(path);
    }

    protected final WebResource.Builder createApplicationBuilder(final String path) {
        return createBuilder("applications/" + path);
    }

    protected final WebResource.Builder createNotificationBuilder(final String path) {
        return createBuilder("notifications/" + path);
    }

    public static class ObjectMapper extends org.codehaus.jackson.map.ObjectMapper {

        public ObjectMapper() {
            super();
            getSerializationConfig().setSerializationInclusion(Inclusion.NON_NULL);
        }

    }

    protected CloudHubException buildException(ClientResponse clientResponse) throws CloudHubException {
        String message = "";
        int status = clientResponse.getStatus();

        if (ClientResponse.Status.BAD_REQUEST.equals(clientResponse.getClientResponseStatus())) {
            message = clientResponse.getEntity(String.class);
        } else {
            message = clientResponse.getClientResponseStatus().getReasonPhrase();
        }
        return new CloudHubException(message, String.valueOf(status));

    }

    @Override
    public String getUrl() {
        return this.url;
    }

    @Override
    public String getUsername() {
        return this.username;
    }

    @Override
    public String getPassword() {
        return this.password;
    }

    @Override
    public String getEnvironmentId() {
        return this.environmentId;
    }
    
    @Override
    public String getAccessToken() {
        return this.accessToken;
    }

    @Override
    public boolean isCsAuthentication() {
        return this.csAuthentication;
    }

}
