package com.mulesoft.cloudhub.client;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.ws.rs.core.MediaType;

import com.mulesoft.ch.rest.model.Application;
import com.mulesoft.ch.rest.model.ApplicationStatus;
import com.mulesoft.ch.rest.model.ApplicationStatusChange;
import com.mulesoft.ch.rest.model.ApplicationUpdateInfo;
import com.mulesoft.ch.rest.model.Deployment;
import com.mulesoft.ch.rest.model.Deployments;
import com.mulesoft.ch.rest.model.Instance;
import com.mulesoft.ch.rest.model.LogLine;
import com.mulesoft.ch.rest.model.LogRequest;
import com.mulesoft.ch.rest.model.LogResults;
import com.mulesoft.ch.rest.model.LogLineWrapperV2;
import com.mulesoft.ch.rest.model.Tenant;
import com.mulesoft.ch.rest.model.TenantResults;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.WebResource;

public class CloudHubDomainConnectionImpl extends CloudHubConnectionImpl implements CloudHubDomainConnectionI {

	private String domain;

	protected CloudHubDomainConnectionImpl(CloudHubConnectionI con, String domain) {
		super(con.getUrl(), con.getUsername(), con.getPassword(), null, false);
		this.domain = domain;
	}

	protected CloudHubDomainConnectionImpl(CloudHubConnectionI con, boolean csAuthentication, String domain) {
		super(con.getUrl(), con.getAccessToken(), con.getEnvironmentId(), false);
		this.domain = domain;
	}

	public CloudHubDomainConnectionImpl(String url, String accessToken, boolean debug) {
		super();
		this.accessToken = accessToken;
		this.init(url, debug);
	}

	public CloudHubDomainConnectionImpl(String url, String accessToken, String environmentId, boolean debug) {
		super();
		this.accessToken = accessToken;
		this.environmentId = environmentId;
		this.init(url, debug);
	}

	@Override
	public String getDomain() {
		return domain;
	}

	@Override
	public Application retrieveApplication() throws CloudHubException {
		ClientResponse clientResponse = createApplicationDomainBuilder("").get(ClientResponse.class);

		if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus()))
			return clientResponse.getEntity(Application.class);

		throw buildException(clientResponse);

	}

	@Override
	/**
	 * Retrieves the application logs. It relies on the app config to get 
	 * the logs from the new or the old logging systems.
	 */
	public LogResults retrieveApplicationLog(Map<String, String> queryParams) throws CloudHubException {
		if (applicationHasNewLogging()) {
			return retrieveApplicationLog_v2(queryParams);
		} else {
			return retrieveApplicationLog_v1(queryParams);
		}
	}

	protected boolean applicationHasNewLogging() throws CloudHubException {
		Application app = retrieveApplication();
		return app.isLoggingNgEnabled();
	}

	public LogResults retrieveApplicationLog_v1(Map<String, String> queryParams) throws CloudHubException {
		WebResource resource = createResource("/applications/" + domain + "/logs");

		if (queryParams != null && !queryParams.isEmpty()) {
			for (String key : queryParams.keySet()) {
				resource = resource.queryParam(key, queryParams.get(key));
			}
		}

		ClientResponse clientResponse = authorizeResource(resource).get(ClientResponse.class);

		if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus()))
			return clientResponse.getEntity(LogResults.class);

		throw buildException(clientResponse);
	}

	public LogResults retrieveApplicationLog_v2(Map<String, String> queryParams) throws CloudHubException {

		Deployment lastDeployment = getLastDeployment();
		ClientResponse clientResponse;
		
		// Checks if search and priority are part of the params map
		if (!queryParams.containsKey("search") && !queryParams.containsKey("priority")) {
			// If not, it uses a GET method to retrieve the logs
			WebResource resource = createResource("/v2/applications/" + domain + "/deployments/" + lastDeployment.getDeploymentId() + "/logs");
			clientResponse = authorizeResource(resource).get(ClientResponse.class);

			if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus()))
				return clientResponse.getEntity(LogResults.class);
		} else {
			// Otherwise, it uses a POST with the parameters
			LogRequest logRequest = new LogRequest();
			if (queryParams.containsKey("search")) {
				logRequest.setText(queryParams.get("search"));
			}
			if (queryParams.containsKey("priority")) {
				logRequest.setPriority(queryParams.get("priority"));
			}
			logRequest.setLimitMsgLen(1000);
			logRequest.setDeploymentId(lastDeployment.getDeploymentId());
			WebResource resource = createResource("/v2/applications/" + domain + "/logs");
			clientResponse = authorizeResource(resource).type(MediaType.APPLICATION_JSON_TYPE).post(ClientResponse.class, logRequest);

			if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus())) {
				LogLineWrapperV2[] logs = clientResponse.getEntity(LogLineWrapperV2[].class);
				LogResults logResults = new LogResults();
				logResults.setData(logsToList(logs));
				logResults.setTotal(new Long(logResults.getData().size()));
				return logResults;
			}
		}

		throw buildException(clientResponse);
	}

	private List<LogLine> logsToList(LogLineWrapperV2[] logs) {
		List<LogLine> logLines = new ArrayList<LogLine>();
		
		for (LogLineWrapperV2 log : logs) {
			logLines.add(log.getEvent());
		}
		
		return logLines;
	}

	public Deployment getLastDeployment() throws CloudHubException {
		WebResource resource = createResource("/v2/applications/" + domain + "/deployments");
		resource = resource.queryParam("orderByDate", "DESC");
		resource = resource.queryParam("loggingVersion", "VERSION_2");

		ClientResponse clientResponse = authorizeResource(resource).get(ClientResponse.class);

		if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus())) {
			Deployments deployments = clientResponse.getEntity(Deployments.class);
			if (deployments.getData().size() > 0) {
				return deployments.getData().get(0);
			}
		}
		throw buildException(clientResponse);
	}

	@Override
	public File downloadApplicationLog(Map<String, String> queryParams) throws CloudHubException {
		if (applicationHasNewLogging()) {
			return downloadApplicationLog_v2(queryParams);
		} else {
			return downloadApplicationLog_v1(queryParams);
		}
	}

	public File downloadApplicationLog_v1(Map<String, String> queryParams) throws CloudHubException {
		WebResource resource = createResource("/applications/" + domain + "/logs/download");
		if (queryParams != null && !queryParams.isEmpty()) {
			for (String key : queryParams.keySet()) {
				resource = resource.queryParam(key, queryParams.get(key));
			}
		}

		ClientResponse clientResponse = authorizeResource(resource).get(ClientResponse.class);

		if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus()))
			return clientResponse.getEntity(File.class);

		throw buildException(clientResponse);
	}

	public File downloadApplicationLog_v2(Map<String, String> queryParams) throws CloudHubException {
		Deployment lastDeployment = getLastDeployment();
		Instance instance = lastDeployment.getInstances().get(0);

		WebResource resource = createResource("/v2/applications/" + domain + "/instances/" + instance.getInstanceId() + "/log-file");
		if (queryParams != null && !queryParams.isEmpty()) {
			for (String key : queryParams.keySet()) {
				resource = resource.queryParam(key, queryParams.get(key));
			}
		}

		ClientResponse clientResponse = authorizeResource(resource).get(ClientResponse.class);

		if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus()))
			return clientResponse.getEntity(File.class);

		throw buildException(clientResponse);
	}

	@Override
	public ApplicationStatus retrieveApplicationStatus() throws CloudHubException {
		ClientResponse clientResponse = createApplicationDomainBuilder("status").get(ClientResponse.class);

		if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus()))
			return ApplicationStatus.valueOf(clientResponse.getEntity(String.class));

		throw buildException(clientResponse);
	}

	@Override
	public void deployApplication(InputStream fileStream, long maxWaitTime) {
		boolean failedBefore = failedBefore();

		ClientResponse clientResponse = createApplicationDomainBuilder("deploy").type(MediaType.APPLICATION_OCTET_STREAM_TYPE).post(ClientResponse.class,
				fileStream);

		if (!ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus()))
			throw buildException(clientResponse);

		waitForFinish(maxWaitTime, failedBefore);
	}

	@Override
	public void deployApplication(File file, long maxWaitTime) {
		try {
			this.deployApplication(new FileInputStream(file), maxWaitTime);
		} catch (FileNotFoundException e) {
			throw new RuntimeException("File does not exist", e);
		}
	}

	@Override
	public Application updateApplication(ApplicationUpdateInfo newApplicationInfo) throws CloudHubException {

		Application application = retrieveApplication();
		ApplicationUpdateInfo oldApplicationInfo = new ApplicationUpdateInfo(application);

		if (!oldApplicationInfo.equals(newApplicationInfo)) {
			ClientResponse clientResponse = createApplicationDomainBuilder("").type(MediaType.APPLICATION_JSON_TYPE).put(ClientResponse.class, newApplicationInfo);

			if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus()))
				return clientResponse.getEntity(Application.class);

			throw buildException(clientResponse);
		}

		return application;

	}

	@Override
	public void updateApplicationStatus(ApplicationStatusChange applicationStatusChange, long maxWaitTime) throws CloudHubException {
		boolean failedBefore = failedBefore();
		ClientResponse clientResponse = createApplicationDomainBuilder("status").type(MediaType.APPLICATION_JSON_TYPE).post(ClientResponse.class,
				applicationStatusChange);

		if (!ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus())) {
			throw buildException(clientResponse);
		}

		waitForFinish(maxWaitTime, failedBefore);
	}

	@Override
	public void deleteApplication() throws CloudHubException {
		ClientResponse clientResponse = createApplicationDomainBuilder().delete(ClientResponse.class);

		if (!ClientResponse.Status.NO_CONTENT.equals(clientResponse.getClientResponseStatus())) {
			throw buildException(clientResponse);
		}

	}

	@Override
	public TenantResults retrieveTenants(Integer limit, Integer offset, String query, Boolean enabled) throws CloudHubException {

		WebResource resource = createResource("/applications/" + domain + "/tenants");

		if (limit != null) {
			resource = resource.queryParam("limit", String.valueOf(limit));
		}

		if (offset != null) {
			resource = resource.queryParam("offset", String.valueOf(offset));
		}

		if (enabled != null) {
			resource = resource.queryParam("enabled", enabled.toString());
		}

		if (query != null) {
			resource = resource.queryParam("search", query);
		}

		ClientResponse clientResponse = authorizeResource(resource).type(MediaType.APPLICATION_JSON_TYPE).get(ClientResponse.class);

		if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus()))
			return clientResponse.getEntity(TenantResults.class);

		throw buildException(clientResponse);

	}

	@Override
	public Tenant retrieveTenant(String tenantId) throws CloudHubException {
		ClientResponse clientResponse = createApplicationDomainTenantBuilder(tenantId).type(MediaType.APPLICATION_JSON_TYPE).get(ClientResponse.class);

		if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus()))
			return clientResponse.getEntity(Tenant.class);

		throw buildException(clientResponse);

	}

	@Override
	public InputStream retrieveTenantClientCertificates(String tenantId) throws CloudHubException {
		ClientResponse clientResponse = createApplicationDomainTenantBuilder(tenantId + "/openVPNConfigs/").type(MediaType.APPLICATION_JSON_TYPE).get(
				ClientResponse.class);
		return clientResponse.getEntity(InputStream.class);
	}

	@Override
	public Tenant createTenant(Tenant tenant) throws CloudHubException {
		ClientResponse clientResponse = createApplicationDomainTenantBuilder("").type(MediaType.APPLICATION_JSON_TYPE).post(ClientResponse.class, tenant);

		if (ClientResponse.Status.CREATED.equals(clientResponse.getClientResponseStatus()))
			return clientResponse.getEntity(Tenant.class);

		throw buildException(clientResponse);

	}

	@Override
	public Tenant updateTenant(Tenant tenant) throws CloudHubException {
		ClientResponse clientResponse = createApplicationDomainTenantBuilder(tenant.getId()).type(MediaType.APPLICATION_JSON_TYPE)
				.put(ClientResponse.class, tenant);

		if (ClientResponse.Status.OK.equals(clientResponse.getClientResponseStatus()))
			return clientResponse.getEntity(Tenant.class);

		throw buildException(clientResponse);
	}

	@Override
	public void deleteTenant(String tenantId) throws CloudHubException {
		ClientResponse clientResponse = createApplicationDomainTenantBuilder(tenantId).type(MediaType.APPLICATION_JSON_TYPE).delete(ClientResponse.class);
		if (!ClientResponse.Status.NO_CONTENT.equals(clientResponse.getClientResponseStatus())) {
			throw buildException(clientResponse);
		}
	}

	@Override
	public void deleteTenants(List<String> tenantIds) throws CloudHubException {
		ClientResponse clientResponse = createApplicationDomainTenantBuilder("").type(MediaType.APPLICATION_JSON_TYPE).delete(ClientResponse.class, tenantIds);
		if (!ClientResponse.Status.NO_CONTENT.equals(clientResponse.getClientResponseStatus())) {
			throw buildException(clientResponse);
		}

	}

	@Override
	public void deleteTenantClientCertificates(String tenantId) throws CloudHubException {
		ClientResponse clientResponse = createApplicationDomainTenantBuilder(tenantId + "/openVPNConfigs/").type(MediaType.APPLICATION_JSON_TYPE).delete(
				ClientResponse.class);
		if (!ClientResponse.Status.NO_CONTENT.equals(clientResponse.getClientResponseStatus())) {
			throw buildException(clientResponse);
		}

	}

	private final WebResource.Builder createApplicationDomainBuilder(final String path) {
		return createBuilder("applications/" + domain + "/" + path);
	}

	private final WebResource.Builder createApplicationDomainBuilder() {
		return createBuilder("applications/" + domain);
	}

	private final WebResource.Builder createApplicationDomainTenantBuilder(final String path) {
		return createBuilder("applications/" + domain + "/tenants/" + path);
	}

	private void waitForFinish(long maxWaitTime, boolean failedBefore) {
		if (maxWaitTime == 0)
			return;
		else if (maxWaitTime == 1) {
			maxWaitTime = Long.MAX_VALUE;
		}

		final long before = System.currentTimeMillis();
		while (System.currentTimeMillis() - before < maxWaitTime) {
			final Application latestApplication = retrieveApplication();
			final ApplicationStatus status = latestApplication.getDeploymentUpdateStatus();

			if (!failedBefore && status != null) {
				if (status.equals(ApplicationStatus.DEPLOY_FAILED)) {
					throw new CloudHubException("Application " + domain + " failed to start.", String.valueOf(ClientResponse.Status.EXPECTATION_FAILED.getStatusCode()));
				}
			}
			try {
				Thread.sleep(1000L);
			} catch (InterruptedException e) {
				// Quit loop when interrupted
				break;
			}
		}

		final Application latestApplication = retrieveApplication();
		final ApplicationStatus status = latestApplication.getDeploymentUpdateStatus();

		if (status == null && latestApplication.getStatus().equals(ApplicationStatus.STARTED))
			return;
		if (status == null && latestApplication.getStatus().equals(ApplicationStatus.UNDEPLOYED))
			return;

		if (status.equals(ApplicationStatus.DEPLOY_FAILED)) {
			throw new CloudHubException("Application " + domain + " failed to start.", String.valueOf(ClientResponse.Status.EXPECTATION_FAILED.getStatusCode()));
		}

		throw new CloudHubException("CloudHub did not finish the process before the specified time", String.valueOf(ClientResponse.Status.GATEWAY_TIMEOUT
				.getStatusCode()));
	}

	private boolean failedBefore() {
		ApplicationStatus status = retrieveApplication().getDeploymentUpdateStatus();
		return status == null ? false : status.equals(ApplicationStatus.DEPLOY_FAILED);
	}

	public static void main(String[] args) {

		// silQAp1
		// String csToken = "77fbd6f6-d9fe-46ad-a228-073ce3812d33";
		// String environmentId = "11935809-b30c-475c-bd28-c6cf00d192f5";

		// datagatewya
		String csToken = "635edb51-ad3f-4ce0-ba6f-0571d198258d";
		String environmentId = "552d191be4b0ccf3b4a11936";

		CloudHubConnectionI ch;
		ch = new CloudHubConnectionImpl("https://anypoint.mulesoft.com/cloudhub/", csToken, environmentId, false);

		Map<String, String> queryParams = new HashMap<String, String>();
		queryParams.put("limit", "100");

		// logs V1
		String domain = "dgw-35f22bf0-d696-4e4c-8e46-9332beb8a422";
		CloudHubDomainConnectionI chd = ch.connectWithDomain(domain);
		LogResults logs = chd.retrieveApplicationLog(queryParams);
		System.out.println("V2 >>> " + logs);
		File logFile = chd.downloadApplicationLog(queryParams);
		System.out.println("V2 FILE >>> " + logFile);
		/*
		 * //logs V2 domain = "dgw-de5504ef-bbed-4194-adee-a5a66910921c"; chd =
		 * ch.connectWithDomain(domain); logs =
		 * chd.retrieveApplicationLog(queryParams); System.out.println("V1 >>> " +
		 * logs); logFile = chd.downloadApplicationLog(queryParams);
		 * System.out.println("V2 FILE >>> " + logFile);
		 */
	}

}
