package com.mulesoft.cloudhub.client;

import java.io.File;
import java.io.InputStream;
import java.util.List;
import java.util.Map;

import com.mulesoft.ch.rest.model.Application;
import com.mulesoft.ch.rest.model.ApplicationStatus;
import com.mulesoft.ch.rest.model.ApplicationStatusChange;
import com.mulesoft.ch.rest.model.ApplicationUpdateInfo;
import com.mulesoft.ch.rest.model.LogResults;
import com.mulesoft.ch.rest.model.Tenant;
import com.mulesoft.ch.rest.model.TenantResults;
import com.mulesoft.cloudhub.client.annotation.Component;
import com.mulesoft.cloudhub.client.annotation.UsedBy;

public interface CloudHubDomainConnectionI extends CloudHubConnectionI {

    /**
     * Returns the connected domain
     *
     * @return the connected domain
     */
    public String getDomain();

    /**
     * Retrieves the application based on the connected domain
     *
     * @return an instance of {@link com.mulesoft.ch.rest.model.Application}
     * @throws CloudHubException if the domain does not exist
     */
    @UsedBy(Component.STUDIO)
    public Application retrieveApplication() throws CloudHubException;

    /**
     * Retrieves the application's status
     *
     * @return one of {@link com.mulesoft.ch.rest.model.ApplicationStatus}
     * @throws CloudHubException if there was an error when retrieving the application status
     */
    public ApplicationStatus retrieveApplicationStatus() throws CloudHubException;

    /**
     * Retrieves the application's logs
     *
     * @param queryParams query parameters
     * @return one of {@link com.mulesoft.ch.rest.model.LogResults}
     * @throws CloudHubException if there was an error when retrieving the application logs
     */
    public LogResults retrieveApplicationLog(Map<String, String> queryParams) throws CloudHubException;

    /**
     * Downloads the application's logs as a file
     *
     * @param queryParams query parameters
     * @return one of {@link java.io.File}
     * @throws CloudHubException if there was an error when retrieving the application logs
     */
    public File downloadApplicationLog(Map<String, String> queryParams) throws CloudHubException;

    /**
     * Deploys an application with the provided fileStream
     *
     * @param fileStream  the stream containing the zip to be deployed
     * @param waitMaxTime Maximum time allowed to deploy/undeploy in miliseconds.
     *                    Case 1: 0, the connector won't wait CH to finish the deploy/undeploy process and will
     *                    return a response based on CH API response status.
     *                    Case 2: 1, the connector will wait until CH finish the deploy/undeploy process and will
     *                    throw an exception if the deploy fails.
     *                    Case 3: anyOthervalue, the connector will wait that amount of miliseconds for CH to finish
     *                    the deploy/undeploy process.
     *                    If takes longer, it will throw an exception. If takes shorter, it will follow the same
     *                    process as case 2.
     * @throws CloudHubException if there was an error while deploying the application
     */
    public void deployApplication(InputStream fileStream, long waitMaxTime) throws CloudHubException;

    /**
     * Deploys an application with the provided fileStream
     *
     * @param file        the file to be deployed
     * @param waitMaxTime Maximum time allowed to deploy/undeploy in miliseconds.
     *                    Case 1: 0, the connector won't wait CH to finish the deploy/undeploy process and will
     *                    return a response based on CH API response status.
     *                    Case 2: 1, the connector will wait until CH finish the deploy/undeploy process and will
     *                    throw an exception if the deploy fails.
     *                    Case 3: anyOthervalue, the connector will wait that amount of miliseconds for CH to finish
     *                    the deploy/undeploy process.
     *                    If takes longer, it will throw an exception. If takes shorter, it will follow the same
     *                    process as case 2.
     * @throws CloudHubException if there was an error while deploying the application
     */
    @UsedBy(value = {Component.STUDIO, Component.DEPLOY_MAVEN_PLUGIN})
    public void deployApplication(File file, long waitMaxTime) throws CloudHubException;

    /**
     * Updates an application's information on your CloudHub account.
     * This method just updates metaData like Mule Version, Description, etc.
     * If you want to update a new stream, use {@link #deployApplication} instead
     *
     * @param application an instance of {@link com.mulesoft.ch.rest.model.ApplicationUpdateInfo} with the
     *                    information to be updated
     * @return an instance of {@link com.mulesoft.ch.rest.model.Application} containing all the new application's
     * information provided by CloudHub
     * @throws CloudHubException if there was an error while updating the application
     */
    @UsedBy(value = {Component.STUDIO, Component.DEPLOY_MAVEN_PLUGIN})
    public Application updateApplication(ApplicationUpdateInfo application) throws CloudHubException;

    /**
     * Updates an application's status based on some specific commands. Can be START or STOP
     *
     * @param applicationStatus an instance of ApplicationStatusChange with the new application status
     * @param waitMaxTime       Maximum time allowed to deploy/undeploy in miliseconds.
     *                          Case 1: 0, the connector won't wait CH to finish the deploy/undeploy process and will
     *                          return a response based on CH API response status.
     *                          Case 2: 1, the connector will wait until CH finish the deploy/undeploy process and
     *                          will throw an exception if the deploy fails.
     *                          Case 3: anyOthervalue, the connector will wait that amount of miliseconds for CH to
     *                          finish the deploy/undeploy process.
     *                          If takes longer, it will throw an exception. If takes shorter, it will follow the
     *                          same process as case 2.
     * @throws CloudHubException if there was an error while updating the application's status.
     */
    @UsedBy(Component.DEPLOY_MAVEN_PLUGIN)
    public void updateApplicationStatus(ApplicationStatusChange applicationStatus, long waitMaxTime) throws
        CloudHubException;

    /**
     * Deletes the application by using the connection's domain.
     * The domain can not be invalid since it was validated when creating the connection.
     *
     * @throws CloudHubException if there was an error while deleting the application
     */
    public void deleteApplication() throws CloudHubException;

    /**
     * Retrieves tenants from the given domain using the given parameters as filters.
     *
     * @param limit   Optional. The maximum number of tenants you want to receive in each call.
     * @param offset  Optional. The off you want to start querying from.
     * @param query   Optional.
     * @param enabled Optional. Flag indicating if you want to retrieve only enabled or disabled
     * @return an instance of {@link com.mulesoft.ch.rest.model.TenantResults}
     * @throws CloudHubException if there was an error while retrieving the tenants
     */
    public TenantResults retrieveTenants(Integer limit, Integer offset, String query, Boolean enabled) throws
        CloudHubException;

    /**
     * Retrieves an specific tenant from the given domain based on the given tenantId
     *
     * @param tenantId The id the tenant you want to retrieve
     * @return an instance of {@link com.mulesoft.ch.rest.model.Tenant} containing all the information provided by
     * CloudHub
     * @throws CloudHubException if there was an error while retrieving the tenant
     */
    public Tenant retrieveTenant(String tenantId) throws CloudHubException;

    /**
     * This method generates a new set of certificates used to establish a VPN connection
     * between the given tenant and the remote location.
     * Take into account that if the tenant already had a set of certificates, using this method will automatically
     * revoke old ones and only those returned by current invocation will work.
     *
     * @param tenantId the id of the tenant you want to generate tenant certificates
     * @return an instance of {@link InputStream}. A zip file containing new certificates
     * @throws CloudHubException if there was an error while generating new certificates
     */
    public InputStream retrieveTenantClientCertificates(String tenantId) throws CloudHubException;

    /**
     * Creates a tenant in the given domain based on the given object
     *
     * @param tenant an instance of {@link com.mulesoft.ch.rest.model.Tenant} with the information
     * @return an instance of {@link com.mulesoft.ch.rest.model.Tenant} containing all the information provided by
     * CloudHub
     * @throws CloudHubException if there was an error while creating the tenant
     */
    public Tenant createTenant(Tenant tenant) throws CloudHubException;

    /**
     * Updates a tenant in the given domain based on the given object
     *
     * @param tenant an instance of {@link com.mulesoft.ch.rest.model.Tenant} with the information
     * @return an instance of {@link com.mulesoft.ch.rest.model.Tenant} containing all the information provided by
     * CloudHub
     * @throws CloudHubException if there was an error while updating the tenant
     */
    public Tenant updateTenant(Tenant tenant) throws CloudHubException;

    /**
     * Deletes a tenant from the given domain based on the given tenant id
     *
     * @param tenantId the id of the tenant you want to delete
     * @throws CloudHubException if there was an error while deleting the tenant
     */
    public void deleteTenant(String tenantId) throws CloudHubException;

    /**
     * Deletes a bunch of tenant from the given domain based on the given tenant ids
     *
     * @param tenantIds the list of tenant's ids you want to delete. IMPORTANT: If null or empty, all tenants will be
     *                  deleted.
     * @throws CloudHubException if there was an error while deleting the tenants
     */
    public void deleteTenants(List<String> tenantIds) throws CloudHubException;

    /**
     * This method revokes current certificates for the given tenant from the given domain
     *
     * @param tenantId the id of the tenant you want to generate tenant certificates
     * @throws CloudHubException if there was an error while deleting certificates
     */
    public void deleteTenantClientCertificates(String tenantId) throws CloudHubException;

}
