package com.mulesoft.cloudhub.client;

import java.util.Collection;
import java.util.List;

import com.mulesoft.ch.rest.model.Account;
import com.mulesoft.ch.rest.model.Application;
import com.mulesoft.ch.rest.model.Notification;
import com.mulesoft.ch.rest.model.NotificationResults;
import com.mulesoft.cloudhub.client.annotation.Component;
import com.mulesoft.cloudhub.client.annotation.UsedBy;

public interface CloudHubConnectionI {

    public String getUrl();

    public String getUsername();

    public String getPassword();

    public Integer getReadTimeout();

    public Integer getConnectionTimeout();

    public String getAccessToken();

    public String getEnvironmentId();

    public boolean isCsAuthentication();

    /**
     * Creates a connection against an specific CloudHub Account
     *
     * @param url               Optional. CloudHub URL. If null, default value is https://cloudhub.io/
     * @param username          CloudHub username to log in
     * @param password          CloudHub password to log in
     * @param sandbox           Optional. Only necessary if you want to connect to the non-default sandbox of the
     *                          account.
     *                          If null, the connection will be established against the default sandbox
     * @param readTimeout       The amount of milliseconds to wait until read timeout
     * @param connectionTimeout The amount of milliseconds to wait until connection timeout
     * @return an instance of {@link com.mulesoft.cloudhub.client.CloudHubConnectionImpl}
     */
    public CloudHubConnectionI createConnection(String url, String username, String password, String sandbox,
                                                Integer readTimeout, Integer connectionTimeout);

    /**
     * Returns a connection against a given domain of the CloudHub account
     *
     * @param domain the domain to which you want to connect
     * @return an instance of {@link com.mulesoft.cloudhub.client.CloudHubDomainConnectionImpl} connected with the
     * specified domain
     */
    @UsedBy(Component.STUDIO)
    public CloudHubDomainConnectionI connectWithDomain(String domain);

    /**
     * Retrieves the CloudHub Account
     *
     * @return an instance of {@link com.mulesoft.ch.rest.model.Account}
     * @throws CloudHubException if there is an error when retrieving the account
     */
    @UsedBy(Component.STUDIO)
    public Account retrieveAccount() throws CloudHubException;

    /**
     * Checks if the domain is valid and available
     *
     * @param domain the domain to be validated
     * @return true if the domain is valid and available to be used. False if the domain is already taken.
     * @throws CloudHubException if there is an error when validating the domain
     */
    @UsedBy(value = {Component.STUDIO, Component.DEPLOY_MAVEN_PLUGIN})
    public boolean isDomainAvailable(String domain) throws CloudHubException;

    /**
     * Retrieves All Applications from an specific CloudHub account
     *
     * @return a collection of {@link com.mulesoft.ch.rest.model.Application}
     * @throws CloudHubException if there is an error when retrieving the applications
     */
    @UsedBy(Component.STUDIO)
    public Collection<Application> retrieveApplications() throws CloudHubException;

    /**
     * Creates an application on your CloudHub account.
     * This method does not deploy your application.
     *
     * @param application an instance of {@link com.mulesoft.ch.rest.model.Application} containing all the
     *                    application's information
     * @return an instance of {@link com.mulesoft.ch.rest.model.Application} containing all the recently created
     * application's information provided by CloudHub
     * @throws CloudHubException if there was an error while creating the application
     */
    @UsedBy(value = {Component.STUDIO, Component.DEPLOY_MAVEN_PLUGIN})
    public Application createApplication(Application application) throws CloudHubException;

    /**
     * Retrieves a bunch of notifications based on given filters
     *
     * @param domain   the domain you want to retrieve the notifications from
     * @param tenantId the id of the tenant you want to retrieve the notifications from
     * @param limit    Optional. The maximum number of notifications you want to retrieve
     * @param offset   Optional. The beginning index from which notifications will be retrieved. For example, if you
     *                 have an offset of 5, and your original
     *                 search returns 10 notifications, only the last 5 notifications will be returned.
     * @param status   Optional. One of {@link com.mulesoft.ch.rest.model.Notification.NotificationStatus}
     * @param message  Optional. If provided, only notifications containing this message in their body will be returned.
     * @return an instance of {@link com.mulesoft.ch.rest.model.NotificationResults}
     * @throws CloudHubException if there was an error while retrieving of filtering the notifications
     */
    public NotificationResults retrieveNotifications(String domain, String tenantId, Integer limit, Integer offset,
                                                     Notification.NotificationStatus status, String message) throws
        CloudHubException;

    /**
     * Retrieves a single notification, based on the given id.
     *
     * @param notificationId The id of the notification you want to retrieve
     * @return an instance of {@link com.mulesoft.ch.rest.model.Notification}
     * @throws CloudHubException if there was an error while retrieving the specified notification
     */
    public Notification retrieveNotification(String notificationId) throws CloudHubException;

    /**
     * Creates a notification based on the given object
     *
     * @param notification an instance of {@link com.mulesoft.ch.rest.model.Notification} you want to create
     * @return an instance of {@link com.mulesoft.ch.rest.model.Notification} with the created notification's
     * information provided by CloudHub
     * @throws CloudHubException if there was an error while creating the notification
     */
    public Notification createNotification(Notification notification) throws CloudHubException;

    /**
     * Updates all notifications status. Can be READ or UNREAD.
     *
     * @param domain    The domain of the notifications you want to update.
     * @param tenantId  The id of the tenant from which the notifications you want to update belong to.
     * @param newStatus one of {@link com.mulesoft.ch.rest.model.Notification.NotificationStatus} representing the
     *                  new status for the notification
     * @throws CloudHubException if there was an error while updating the notification status
     */
    public void updateNotificationsStatus(String domain, String tenantId, Notification.NotificationStatus newStatus)
        throws CloudHubException;

    /**
     * Updates a notification status. Can be READ or UNREAD.
     *
     * @param notificationId The id of the notification you want to update
     * @param newStatus      one of {@link com.mulesoft.ch.rest.model.Notification.NotificationStatus} representing
     *                       the new status for the notification
     * @throws CloudHubException if there was an error while updating the notification status
     */
    public void updateNotificationStatus(String notificationId, Notification.NotificationStatus newStatus) throws
        CloudHubException;

    /**
     * This method returns all the supported mule versions by CloudHub
     *
     * @return a list of strings, with all the supported mule versions by CloudHub
     * @throws CloudHubException
     */
    @UsedBy(Component.STUDIO)
    public List<String> getSupportedMuleVersions() throws CloudHubException;
}
