/*
 * Copyright (c) 2025, Salesforce, Inc.,
 * All rights reserved.
 * For full license text, see the LICENSE.txt file
 */
%dw 2.7
import * from com::mulesoft::connectivity::transport::Http
import * from com::mulesoft::connectivity::Model
import Int32, Int64 from com::mulesoft::connectivity::Types
import DefaultValue, Hidden from com::mulesoft::connectivity::decorator::Annotations
import Secure from com::mulesoft::connectivity::datacloud::Metadata

/**
 * Enumeration of DataCloud connector framework features that can be enabled for connectors.
 * 
 * These features control the capabilities and behavior of DataCloud connectors, determining
 * which functionality is available and how the connector integrates with the DataCloud platform.
 * Connectors can support multiple features simultaneously.
 */
type DCConnectorFrameworkFeature =
      "INGRESS" | // All connectors that build for ingress should have it
      "EGRESS" | // All connectors that build for egress should have it
      "BATCH" | // All connectors that build for Bring Your own Lake should have it
      "BYOL" | // All connectors that build for Bring Your own Lake should have it
      "BYOM" | // All connectors that build for Bring Your own Model should have it
      "FILEBASED" | // File Based connectors should have it like S3, SFTP etc
      "OBJECTBASED" | // Object Based connectors should have it like Relation databases, NoSQL dbs
      "UNSTRUCTURED" | // All connectors that build for unstructured should have it
      "NOCONNECTION" | // To hide connection from user experience for programmatically created connectors
      "INCREMENTAL_UPDATE" | // All connectors that build for ML should have it - An incremental update feature for Ingress connectors
      "STREAMING" | // All connector that build for streaming should have it
      "STATIC_SCHEMA" | // Connectors with static schema provider by user should have it
      "UNIQUE_CONNECTION" | // To restrict creation of duplicate connectors on one tenant
      "WDC_CONNECTOR" | // Label a connector as WDC Connector
      "FAST_INGEST" |
      "UNKNOWN"

/**
 * Time unit enumeration for configuring intervals and timeouts in rate limiting and retry configurations.
 */
type TimeUnit = "MILLISECONDS" | "SECONDS" | "MINUTES" | "HOURS" | "DAYS"

/**
 * Configuration object for rate limiting behavior in DataCloud operations.
 * 
 * This type defines how rate limiting should be applied to connector operations,
 * including dynamic ID resolution, rate thresholds, and request handling behavior.
 * Rate limiting can be scoped to tenants and configured with custom timeout behavior.
 */
type RateLimiterConfig = {|
    idProvider?: {|  // Dynamic id provider that resolves at runtime
        inputSelectors?: Array<{
            key: String, // The selector key
            secured?: Boolean // Whether the value is sensitive and has to be secured (False by default)
        }>, // Field selectors where to extract the value from operation/connection's input parameters
        additionalKeyValues?: Dictionary<String> // Additional key-value pairs to be included as identifiers of the rate limiter
    |},
    enabled?: Boolean, // Whether the limiter is active (True by default)
    tenantScoped?: Boolean, // Whether the limiter is DC tenant scoped (True by default)
    rate: {|
        value: Int64, // Rate value
        interval: Int64, // Rate interval value
        intervalUnit: TimeUnit // Rate interval unit
    |},
    request?: {|
        permits?: Int32, // The cost of the invocation (1 by default)
        timeout?: Int64, // timeout value (1 by default)
        timeoutUnit?: TimeUnit // Timeout unit ("HOURS" by default)
    |}
|}


type JWT_BEARER = { protocolType : @DefaultValue(value = "JWT_BEARER") @Hidden() @Secure() String }
type OAUTH_CC = { protocolType : @DefaultValue(value = "OAUTH_CC") @Hidden() @Secure() String }
type OAUTH_AC = { protocolType : @DefaultValue(value = "OAUTH_AC") @Hidden() @Secure() String }

/**
 * Comprehensive type definition for DataCloud objects.
 */
type DCObject<InputType <: Object, NextPageInputType <: Object, ResultType, ItemType, ErrorType <: ResultFailure, ConnectionInputAttributes <: Object, ConnectionType <: HttpConnection, TriggerInputType, TriggerResultItemType, OperationResultItemType, Watermark, ContentProviderInput <: Object>  = {|
  name: String,
  'type'?: "USER" | "GROUP" | "PERMISSION" | "COMMENT" | "ACTIVITY" | "STRUCTURED_DATA" | "UNSTRUCTURED_DATA",
  resolvedBy: (Operation<InputType, ResultType, ErrorType, ConnectionType> | PaginatedOperation<InputType, NextPageInputType, Page<ItemType,InputType>, ErrorType, ConnectionType> | Trigger<TriggerInputType, InputType, ResultType, NextPageInputType, ErrorType, ConnectionType, TriggerResultItemType, OperationResultItemType, Watermark>) & {|
        rateLimiterConfigs?: Dictionary<RateLimiterConfig>
    |},
  connectionProvider?: HttpConnectionProvider<ConnectionInputAttributes, HttpAuthenticationType>,
  contentProvider?: Operation<ContentProviderInput, Binary, ErrorType, ConnectionType> & {|
        rateLimiterConfigs?: Dictionary<RateLimiterConfig>
    |},
  navigableContainer: {
    isContainer: (ItemType) -> Boolean,
    binding: (ItemType) -> InputType,
    doProcess?: Boolean
  },
  isHidden?: Boolean,
  customHandler?: "IMPERSONATION"
|}

/**
 * Extended DCObject type that supports dependency relationships with parent objects.
 */
type DependentDCObject<InputType <: Object, NextPageInputType <: Object, ResultType, ItemType, ErrorType <: ResultFailure, ConnectionInputAttributes <: Object, ConnectionType <: HttpConnection, ParentInputType <: Object, ParentNextPageInputType <: Object, ParentResultType, ParentItemType, TriggerInputType, TriggerResultItemType, OperationResultItemType, Watermark, ContentProviderInput <: Object> =
  DCObject<InputType, NextPageInputType, ResultType, ItemType, ErrorType, ConnectionInputAttributes, ConnectionType, TriggerInputType, TriggerResultItemType, OperationResultItemType, Watermark, ContentProviderInput>
  & {|
    dependsOn?: {
      object: DCObject<ParentInputType, ParentNextPageInputType, ParentResultType, ParentItemType, ErrorType, ConnectionInputAttributes, ConnectionType, TriggerInputType, TriggerResultItemType, OperationResultItemType, Watermark, ContentProviderInput>,
      binding: (ParentItemType) -> InputType,
      when?: (InputType) -> Boolean
    }
  |}

/**
 * Complete definition of a DataCloud connector including metadata, features, connections, and objects.
 */
type DataCloudConnector<InputAttributes <: Object, HttpAuthenticationType> = {|
    name: String, // White spaces disallowed
    label: String,
    description: String,
    displayName: String,
    vendor: String,
    version: String,
    ingressDescription?: String,
    egressDescription?: String,
    releaseLevel: "ACCESS_CHECK" | "IN_DEV"| "PILOT" | "BETA" | "GA",
    accessCheck?: String,
    features: Array<DCConnectorFrameworkFeature>,
    featuresConfig?: {|
        INGRESS?: {|
            ACCESS_CHECK?: String, // To hide feature behind the gate
            FREQUENT_INGEST?: Boolean // To add 5, 15, 30 minute schedule to connector. Q4 team should give clearance first
        |},
        EGRESS?: {|
            ACCESS_CHECK: String // To hide feature behind the gate
        |},
        BATCH?: {|
            ACCESS_CHECK: String // To hide feature behind the gate
        |},
        BYOL?: {|
            ACCESS_CHECK: String // To hide feature behind the gate
        |},
        BYOM?: {|
            ACCESS_CHECK: String // To hide feature behind the gate
        |},
        FILEBASED?: {|
            SUPPORT_PARQUET_FILE_TYPE?: "true" | "false", // For connectors that support Parquet
            SUPPORT_ENHANCED_REFRESH?: "true" | "false", // For connectors that support Enhanced Refresh(Partial refresh and full refresh options)
            SUPPORT_DATASYNC?: "true" | "false", // For AWS datasync connector
            CUSTOM_FILE_SELECTOR?: String, // For File Based connectors which use a custom file selector, specifies the custom selector fields
            OBJECT_NAME?: String, // For File Based connectors which use a custom file selector, specifies the object name to use
            FILE_UPLOAD?: "true" | "false"  // For File Upload connector
        |},
        OBJECTBASED: {|
            THREE_LEVEL_SELECTION?: "true" | "false", // For connectors that support 3 level selection(Schema and Database are selected on data stream level)
            TWO_LEVEL_SELECTION?: "true" | "false", // For connectors that support 2 level selection(Schema is selected on data stream level)
            DISABLE_FIELD_TYPE_CHANGE?: "true" | "false" // For Object Based connectors that does not allow change field types
        |},
        UNSTRUCTURED?: {|
            ACCESS_CHECK?: String, // To hide feature behind the gate
            DCF_INCREMENTAL?: "true" | "false" // All connectors that build for unstructured and support sending incremental data should have
        |},
        NOCONNECTION?: {|
            READ_ONLY: "true" | "false" // To make connection readonly
        |},
        INCREMENTAL_UPDATE: {|
            ACCESS_CHECK?: String, // To hide feature behind the gate
            DELETE_SUPPORT?: "true" | "false" // An additional feature to support deletion of data lake row
        |},
        UNIQUE_CONNECTION: {|
            UNIQUE_ATTRIBUTES: String // A list of attributes that will be used to find duplicates
        |}
    |},
    customErrorLabels?: Dictionary<String>,
    connection?: {
        provider: HttpConnectionProvider<InputAttributes, HttpAuthenticationType>,
        rateLimiterConfigs?: Dictionary<RateLimiterConfig>,
    },
    connections?: Dictionary<{
        provider: HttpConnectionProvider<InputAttributes, HttpAuthenticationType>,
        rateLimiterConfigs?: Dictionary<RateLimiterConfig>,
    }>,
    testConnection: HttpTestConnection,
    objects: Dictionary<DependentDCObject>,
    externalCallout: Dictionary<DependentDCObject>
|}
