/*
 * Copyright (c) 2025, Salesforce, Inc.,
 * All rights reserved.
 * For full license text, see the LICENSE.txt file
 */
%dw 2.7
import * from com::mulesoft::connectivity::Model
import * from com::mulesoft::connectivity::decorator::Executor
import * from com::mulesoft::connectivity::decorator::Operation
import * from com::mulesoft::connectivity::transport::Http

/**
* Transform an operation into a offset paginated one specifying how to handle the offset parameter
* and a condition on the result to check if there is a next page.
* This is useful for complex responses where the actual result is nested inside the response.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `operation` | Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType&#62; | 
* | `itemsResolver` | &#40;OpResultType&#41; &#45;&#62; Array<ItemType&#62; | 
* | `paramToOffsetFn` | &#40;OpParam&#41; &#45;&#62; Number | 
* | `offsetToParamFn` | &#40;OpParam, Number&#41; &#45;&#62; OpParam | 
* | `nextPageCondition` | &#40;OpResultType&#41; &#45;&#62; Boolean | 
* |===
*
*/
fun customOffsetPaginated<OpParam, OpResultType, OpResultErrorType<: ResultFailure, OpConnectionType, ItemType>(
    operation: Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType>,
    itemsResolver: (OpResultType) -> Array<ItemType>,
    paramToOffsetFn: (OpParam) -> Number,
    offsetToParamFn: (OpParam, Number) -> OpParam,
    nextPageCondition: (OpResultType) -> Boolean
): PaginatedOperation<OpParam, OpParam, Page<ItemType, OpParam>, OpResultErrorType, OpConnectionType> =
      (operation paginated (param, page) -> do {
         var items = itemsResolver(page)
         ---
         {
           items: items,
           (nextPage: {
                  args: offsetToParamFn(param, paramToOffsetFn(param) + sizeOf(items))
                 }) if nextPageCondition(page)
         }
      })
