/*
 * Copyright (c) 2025, Salesforce, Inc.,
 * All rights reserved.
 * For full license text, see the LICENSE.txt file
 */
%dw 2.8
import dw::Crypto
import * from dw::core::Binaries


/**
 * Generates a unique string identifier based on the provided arguments using SHA-256 hashing.
 * 
 * This function creates a deterministic hash from an array of values, making it useful for
 * generating consistent identifiers for caching, deduplication, or correlation purposes.
 * The function handles null values gracefully by converting them to the string "null".
 * 
 * === Parameters
 * 
 * [%header, cols="1,1,3"]
 * |===
 * | Name | Type | Description
 * | `args` | Array<Any> | Array of values to include in the hash computation
 * |===
 * 
 * === Returns
 * 
 * A hexadecimal string representation of the SHA-256 hash of the concatenated arguments.
 * 
 * === Example
 * 
 * ----
 * generateUniqueString(["user", 123, "session"]) 
 * // Returns: "a1b2c3d4e5f6..." (example hash)
 * ----
 * 
 * @param args the array of arguments to hash
 * @return hexadecimal string representation of the SHA-256 hash
 */
fun generateUniqueString(args: Array<Any>): String =
 toHex(Crypto::hashWith( args map ((item, index) -> if (item == null) "null" else item as String) joinBy "::" as Binary, "SHA-256"))
