/*
 * Copyright (c) 2025, Salesforce, Inc.,
 * All rights reserved.
 * For full license text, see the LICENSE.txt file
 */
%dw 2.8

import * from com::mulesoft::connectivity::Model
import HttpOperation, HttpResponse from com::mulesoft::connectivity::transport::Http


/**
* Handles HTTP errors by adding retry-related categories for specific HTTP status codes.
* This function checks if the failure contains an HttpResponse and only processes HTTP-specific errors.
* For non-HTTP failures, it returns the original failure unchanged.
* 
* For HTTP failures with the following status codes, it adds specific categories:
* - 401 (Unauthorized) - adds CLIENT_ERROR, AUTHENTICATION, RECONNECT categories
* - 429 (Too Many Requests) - adds CLIENT_ERROR, THROTTLING, RETRY categories  
* - Any 5XX status code (500-599) - adds SERVER_ERROR, RETRY categories
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `failureValue` | FailureResultType | A failure result object that may contain an HttpResponse error value
* | `inputValue` | InputType | The original input value that caused the failure
* |===
*
* === Example
*
* This example shows how the `handleError` function behaves with HTTP and non-HTTP failures.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.7
* import handleError from Http
* ---
* {
*   retryable429: handleError({
*     error: {
*       value: { status: 429 },
*       categories: ["HTTP_ERROR"],
*       kind: "HTTP_FAILURE"
*     }
*   }, {}).error.categories,
*   nonRetryable404: handleError({
*     error: {
*       value: { status: 404 },
*       categories: ["HTTP_ERROR"],
*       kind: "HTTP_FAILURE"
*     }
*   }, {}).error.categories,
*   retryable500: handleError({
*     error: {
*       value: { status: 500 },
*       categories: ["HTTP_ERROR"],
*       kind: "HTTP_FAILURE"
*     }
*   }, {}).error.categories,
*   nonHttpError: handleError({
*     error: {
*       value: { message: "Database connection failed" },
*       categories: ["DATABASE_ERROR"],
*       kind: "CONNECTION_FAILURE"
*     }
*   }, {}).error.categories
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "retryable429": ["HTTP_ERROR", "CLIENT_ERROR", "THROTTLING", "RETRY"],
*   "nonRetryable404": ["HTTP_ERROR"],
*   "retryable500": ["HTTP_ERROR", "SERVER_ERROR", "RETRY"],
*   "nonHttpError": ["DATABASE_ERROR"]
* }
* ----
*
*/
fun handleError<InputType <: Object, FailureBodyType>(failureValue: ResultFailure<HttpResponse<FailureBodyType>, Error>, inputValue: InputType): ResultFailure<HttpResponse<FailureBodyType>, Error> =
    do {
        var fixedRetryableErrors = [401,429]
        var httpResponse = failureValue.error.value as HttpResponse
        var httpStatusCode = httpResponse.status default -1
        ---
        if ((fixedRetryableErrors contains httpStatusCode) or (httpStatusCode >= 500 and httpStatusCode < 600))
            failure(
                httpResponse,
                {
                    kind: failureValue.error.kind!,
                    categories: getRetryableErrorCategories(failureValue.error.categories!, httpResponse)
                },
                failureValue.error.description
            )
        else failureValue
    }

/**
* Adds appropriate retry-related error categories based on HTTP status code.
* This function determines which additional categories should be added to existing 
* error categories based on the HTTP response status code.
* Adjusting throttling status codes according to https://datatracker.ietf.org/doc/draft-ietf-httpapi-ratelimit-headers/.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `categories` | Array<Category> | Existing error categories to which new categories will be added
* | `httpResponse` | HttpResponse | HTTP response object containing the status code
* |===
*
* === Category Mappings
*
* [%header, cols="1,3"]
* |===
* | Status Code | Added Categories
* | 401 | CLIENT_ERROR, AUTHENTICATION, RECONNECT
* | 429 | CLIENT_ERROR, THROTTLING, RETRY
* | 503 | SERVER_ERROR, THROTTLING, RETRY
* | 5XX (500-599, except 503) | SERVER_ERROR, RETRY
* |===
*
* === Example
*
* This example shows how different HTTP status codes result in different category combinations.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.7
* import getRetryableErrorCategories from Http
* ---
* {
*   unauthorized: getRetryableErrorCategories(["HTTP_ERROR"], { status: 401 }),
*   throttled: getRetryableErrorCategories(["HTTP_ERROR", "RATE_LIMIT"], { status: 429 }),
*   serviceUnavailable: getRetryableErrorCategories(["HTTP_ERROR"], { status: 503 }),
*   serverError: getRetryableErrorCategories(["HTTP_ERROR"], { status: 500 }),
*   duplicateHandling: getRetryableErrorCategories(["HTTP_ERROR", "RETRY"], { status: 429 })
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "unauthorized": ["HTTP_ERROR", "CLIENT_ERROR", "AUTHENTICATION", "RECONNECT"],
*   "throttled": ["HTTP_ERROR", "RATE_LIMIT", "CLIENT_ERROR", "THROTTLING", "RETRY"],
*   "serviceUnavailable": ["HTTP_ERROR", "SERVER_ERROR", "THROTTLING", "RETRY"],
*   "serverError": ["HTTP_ERROR", "SERVER_ERROR", "RETRY"],
*   "duplicateHandling": ["HTTP_ERROR", "RETRY", "CLIENT_ERROR", "THROTTLING"]
* }
* ----
*
*/
fun getRetryableErrorCategories<Category <: String>(categories: Array<Category>, httpResponse: HttpResponse) = do {
    var additionalCategories = httpResponse.status  match {
   	    case 401 -> ["CLIENT_ERROR","AUTHENTICATION","RECONNECT"]
    	case 429 -> ["CLIENT_ERROR","THROTTLING","RETRY"]
    	case 503 -> ["SERVER_ERROR","THROTTLING","RETRY"]
    	case status if status >=500 and status < 600 -> ["SERVER_ERROR","RETRY"]
    	else -> []
    }
    ---
    (categories ++ additionalCategories) distinctBy (category) -> category
}



/**
* Transforms an HTTP operation by applying input and/or error transformations while preserving the original result type.
* This version focuses on adapting inputs and handling errors without changing the successful response structure.
* The `outFailure` transformer is always available for custom error handling.
*
* === When to Use This Version
*
* Use this version when you need to:
* - Transform input parameters to match what the original operation expects
* - Add custom error handling logic  
* - Keep the original successful response structure unchanged
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `operation` | HttpOperation<OldIn, ResultType, FailureBodyType, Conn> | The original HTTP operation to transform
* | `transformer.in` | (NewIn) -> OldIn | _(Optional)_ Function to transform new input format to the original operation's expected input format
* | `transformer.outFailure` | (ResultFailure, NewIn) -> ResultFailure | _(Optional)_ Function to customize error handling - result still passed through `handleError`
* |===
*
* === Returns
*
* `HttpOperation<NewIn, ResultType, FailureBodyType, Conn>` - The transformed operation with new input type but **same result type as original**.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
*
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
*
* ----
*
*/
fun mapOperation<OldIn <: Object, FailureBodyType, Conn, NewIn <: Object, ResultType>(
    operation: HttpOperation<OldIn, ResultType, FailureBodyType, Conn>,
    transformer: {|
      in?: (NewIn) -> OldIn,
      outFailure?: (ResultFailure<HttpResponse<FailureBodyType>, Error>, NewIn) -> ResultFailure<HttpResponse<FailureBodyType>, Error>,
    |}
): HttpOperation<NewIn, ResultType, FailureBodyType, Conn> = do {
    var actualInputFunction = if (transformer.in?) (v) -> transformer.in(v) else (v) -> v
    var actualResultFailureFunction = if (transformer.outFailure?) (v, i) -> handleError(transformer.outFailure(v,i),i) else (v, i) -> handleError(v,i)
    ---
    operation update {
       case .executor -> (v: NewIn, c : Conn) ->
           operation.executor(actualInputFunction(v), c) match {
               case is ResultSuccess<HttpResponse<ResultType>> -> $
               case is ResultFailure<HttpResponse<FailureBodyType>, Error> -> actualResultFailureFunction($,v)
           }
    }
}


/**
* Transforms an HTTP operation by applying input, output, and/or error transformations with a **well-known result type**.
* This version allows you to transform both the input format and the successful response structure, returning 
* a specific result type. The `outFailure` transformer is always available for custom error handling.
*
* === When to Use This Version
*
* Use this version when you need to:
* - Transform successful operation results to a specific, known type
* - Enhance or restructure the response data  
* - Optionally transform input parameters
* - Add custom error handling logic
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `operation` | HttpOperation<OldIn, ResultType, FailureBodyType, Conn> | The original HTTP operation to transform
* | `transformer.in` | (NewIn) -> OldIn | _(Optional)_ Function to transform new input format to the original operation's expected input format
* | `transformer.outSuccess` | (ResultType, NewIn) -> NewResultType | *Required* - Function to transform successful results to a specific new type
* | `transformer.outFailure` | (ResultFailure, NewIn) -> ResultFailure | _(Optional)_ Function to customize error handling - result still passed through `handleError`
* |===
*
* === Returns
*
* `HttpOperation<NewIn, NewResultType, FailureBodyType, Conn>` - The transformed operation with **well-known result type `NewResultType`**.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
*
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
*
* ----
*
*/
fun mapOperation<OldIn <: Object, FailureBodyType, Conn, NewIn <: Object, ResultType, NewResultType>(
    operation: HttpOperation<OldIn, ResultType, FailureBodyType, Conn>,
    transformer: {|
      in?: (NewIn) -> OldIn,
      outSuccess: (ResultType, NewIn) -> NewResultType,
      outFailure?: (ResultFailure<HttpResponse<FailureBodyType>, Error>, NewIn) -> ResultFailure<HttpResponse<FailureBodyType>, Error>,
    |}
): HttpOperation<NewIn, NewResultType, FailureBodyType, Conn> = do {
    var actualInputFunction = if (transformer.in?) (v) -> transformer.in(v) else (v) -> v
    var actualResultSuccessFunction = (v, i) -> transformer.outSuccess(v, i)
    var actualResultFailureFunction = if (transformer.outFailure?) (v, i) -> handleError(transformer.outFailure(v,i),i) else (v, i) -> handleError(v,i)
    ---
    operation update {
       case .executor -> (v: NewIn, c : Conn) ->
           operation.executor(actualInputFunction(v), c) match {
               case is ResultSuccess<HttpResponse<ResultType>> -> $ update {
                    case body at .value.body -> actualResultSuccessFunction(body, v)
               }
               case is ResultFailure<HttpResponse<FailureBodyType>, Error> -> actualResultFailureFunction($,v)
           }
    }
}

/**
* Transforms a paginated operation by applying input and/or error transformations while preserving the original page item type.
* This version focuses on adapting inputs and handling errors without changing the structure of the items in each page.
* The `outFailure` transformer is always available for custom error handling, and it works for both initial page 
* requests and subsequent next-page requests.
*
* === When to Use This Version
*
* Use this version when you need to:
* - Transform input parameters to match what the original paginated operation expects
* - Add custom error handling logic that applies to all pagination requests  
* - Keep the original page item structure unchanged
* - Handle both initial page and next page requests consistently
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `operation` | PaginatedOperation<OldIn, NextIn, Page<OldOut, NextIn>, FailureResultType, Conn> | The original paginated operation to transform
* | `transformer.in` | (NewIn) -> OldIn | _(Optional)_ Function to transform new input format to the original operation's expected input format
* | `transformer.outFailure` | (FailureResultType, NewIn &#124; NextIn) -> ResultFailure | _(Optional)_ Function to customize error handling - applies to both initial and next page requests
* |===
*
* === Returns
*
* `PaginatedOperation<NewIn, NextIn, Page<OldOut, NextIn>, ResultFailure, Conn>` - The transformed operation with new input type but **same page item type as original**.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
*
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
*
* ----
*
*/
fun mapPaginatedOperation<OldIn <: Object, NextIn <: Object, OldOut, FailureResultType <: ResultFailure, Conn, NewIn <: Object, FailureBodyType>(
    operation: PaginatedOperation<OldIn, NextIn, Page<OldOut, NextIn>, FailureResultType, Conn>,
    transformer: {|
      in?: (NewIn) -> OldIn,
      outFailure?: (FailureResultType, NewIn | NextIn) -> ResultFailure<HttpResponse<FailureBodyType>, Error>,
    |}
): PaginatedOperation<NewIn, NextIn, Page<OldOut, NextIn>, ResultFailure, Conn> = do {
    var actualInputFunction = if (transformer.in?) (v) -> transformer.in(v) else (v) -> v
    var actualResultFailureFunction = if (transformer.outFailure?) (v, i) -> handleError(transformer.outFailure(v,i),i) else (v, i) -> handleError(v,i)
    ---
    operation update {
          case .executor -> (v: NewIn, c : Conn) ->
              operation.executor(actualInputFunction(v), c) match {
                  case is ResultSuccess<Page<OldOut, NextIn>> -> $
                  case is FailureResultType -> actualResultFailureFunction($,v)
              }
          case .nextPageExecutor -> (v : NextIn, c : Conn) ->
              operation.nextPageExecutor(v, c) match {
                  case is ResultSuccess<Page<OldOut, NextIn>> -> $
                  case is FailureResultType -> actualResultFailureFunction($,v)
              }
     }
}

/**
* Transforms a paginated operation by applying input, output, and/or error transformations with **well-known page item types**.
* This version allows you to transform both the input format and the structure of items in each page, returning 
* a specific page item type. The `outFailure` transformer is always available for custom error handling
* and works for both initial page requests and subsequent next-page requests.
*
* === When to Use This Version  
*
* Use this version when you need to:
* - Transform page items to a specific, known structure across all pages
* - Enhance, enrich, or restructure the data in paginated responses  
* - Optionally transform input parameters
* - Add custom error handling logic for all pagination requests
* - Maintain consistent data transformation across initial and next page results
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `operation` | PaginatedOperation<OldIn, NextIn, Page<OldOut, NextIn>, FailureResultType, Conn> | The original paginated operation to transform
* | `transformer.in` | (NewIn) -> OldIn | _(Optional)_ Function to transform new input format to the original operation's expected input format
* | `transformer.outSuccess` | (Array<OldOut>, NewIn &#124; NextIn) -> Array<NewOut> | *Required* - Function to transform page items to a specific new type for all pages
* | `transformer.outFailure` | (FailureResultType, NewIn &#124; NextIn) -> ResultFailure | _(Optional)_ Function to customize error handling - applies to both initial and next page requests
* |===
*
* === Returns
*
* `PaginatedOperation<NewIn, NextIn, Page<NewOut, NextIn>, ResultFailure, Conn>` - The transformed operation with **well-known page item type `NewOut`**.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
*
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
*
* ----
*
*/
fun mapPaginatedOperation<OldIn <: Object, NextIn <: Object, OldOut, FailureResultType <: ResultFailure, Conn, NewIn <: Object, NewOut, FailureBodyType>(
    operation: PaginatedOperation<OldIn, NextIn, Page<OldOut, NextIn>, FailureResultType, Conn>,
    transformer: {|
      in?: (NewIn) -> OldIn,
      outSuccess: (Array<OldOut>, NewIn | NextIn) -> Array<NewOut>,
      outFailure?: (FailureResultType, NewIn | NextIn) -> ResultFailure<HttpResponse<FailureBodyType>, Error>,
    |}
): PaginatedOperation<NewIn, NextIn, Page<NewOut, NextIn>, ResultFailure, Conn> = do {
    var actualInputFunction = if (transformer.in?) (v) -> transformer.in(v) else (v) -> v
    var actualResultSuccessFunction = (v, i) -> v update { case value at .value.items -> transformer.outSuccess(value, i)}
    var actualResultFailureFunction = if (transformer.outFailure?) (v, i) -> handleError(transformer.outFailure(v,i),i) else (v, i) -> handleError(v,i)
    ---
    operation update {
          case .executor -> (v: NewIn, c : Conn) ->
              operation.executor(actualInputFunction(v), c) match {
                  case is ResultSuccess<Page<OldOut, NextIn>> -> actualResultSuccessFunction($,v)
                  case is FailureResultType -> actualResultFailureFunction($,v)
              }
          case .nextPageExecutor -> (v : NextIn, c : Conn) ->
              operation.nextPageExecutor(v, c) match {
                  case is ResultSuccess<Page<OldOut, NextIn>> -> actualResultSuccessFunction($,v)
                  case is FailureResultType -> actualResultFailureFunction($,v)
              }
     }
}
