%dw 2.8

import * from com::mulesoft::connectivity::Model
import * from com::mulesoft::connectivity::transport::Http
import * from com::mulesoft::connectivity::decorator::Annotations
import * from com::mulesoft::connectivity::decorator::PaginationStrategies
import * from com::mulesoft::connectivity::flow::Annotations
import * from com::mulesoft::connectivity::flow::QueryBuilder
import * from com::mulesoft::connectivity::decorator::Trigger


// ----------------
// -- Connection --
// ----------------

type AuthConnection = {
    orgId: String,
    token: String
}

var bearerConnection = defineBearerHttpConnectionProvider<AuthConnection>(
    (parameter) -> { token: parameter.token },
    (parameter) -> {
        baseUri: "http://localhost"
    }
)

var getAccountMetadataForTestConnection : Operation<{}, Any, ResultFailure<Any, Error>, HttpConnection> = {
    name: "getAccountMetadataForTestConnection",
    displayName: "Get Account Metadata For TestConnection",
    executor: (parameter, connection) -> do {
        var response = connection({
            method: 'GET',
            path: '/sobjects/Account/describe'
        })
        ---
        if (response.status == 200)
            success(response)
        else
            failure(response)
    }
}

var testConnection : HttpTestConnection = {
    validate: defineTestConnection(
        getAccountMetadataForTestConnection,
        (response) -> {
            isValid: response.success,
            message: if (response.success) "Connection test succeeded" else "Connection test failed"
        }
    )
}

// ----------------
// -- Query Builder --
// ----------------

fun getFieldSelection(fieldsSelection: AllFieldsSelection): String = "fields(ALL)"
fun getFieldSelection(fieldsSelection: CustomFieldsSelection): String = fieldsSelection.fields joinBy ", "

fun getValueStr(val: String) : String = "'"++ val ++ "'"
fun getValueStr(val: Number) : String = val as String
fun getValueStr(val: Null) : String = "null"
fun getValueStr(val: Array<String>) : String = "(" ++ (val map ((item) -> "'" ++ item ++ "'") joinBy ", ")  ++ ")"

fun getOperator(val: String) : String = val match {
  case "equals" -> "="
  case "isNotEqual" -> "!="
  case "lessThan" -> "<"
  case "lessThanOrEquals" -> "<="
  case "greaterThan" -> ">"
  case "greaterThanOrEquals" -> ">="
  case "isNull" -> ""
  case "in" -> "in"
  case "between" -> "between"
}

fun getCondition(condition: QueryCondition): String = do {
    var left = condition.field ++ " " ++ getOperator(condition.operator)
    var right = if (condition.operator == 'between')
                       " " ++ (condition.values as Array)[0] as String ++ " and " ++ (condition.values as Array)[1] as String
                     else if (condition.operator == 'isNull')
                        if (condition.value)
                            "= null"
                        else
                            "!= null"
                     else
                       if (condition.value?)
                         " " ++ (getValueStr(condition.value))
                       else
                         ""
    ---
    left ++ right
}

fun getFilter(filter: NoQueryFilter): String =  ""
fun getFilter(filter: AndQueryFilter): String =  if (isEmpty(filter.conditions)) "" else " WHERE " ++ (filter.conditions map getCondition($) joinBy " AND ")
fun getFilter(filter: OrQueryFilter): String =  if (isEmpty(filter.conditions)) "" else " WHERE " ++ (filter.conditions map getCondition($) joinBy " OR ")
fun getFilter(filter: CustomQueryFilter): String = if (isEmpty(filter.conditions)) "" else " WHERE " ++ filter.expression replace /[0-9]+/ with getCondition(filter.conditions[$[0]-1]!)

fun getLimit(limit: Number | Null = 200): String = " LIMIT " ++ limit as String
fun getOffset(offset: Number | Null = 0): String = " OFFSET " ++ offset as String

fun getFieldOrder(fieldOrder: QueryFieldOrderPair): String = fieldOrder.field ++ " " ++ fieldOrder.order
fun getOrderBy(orderBy: QueryFieldNoOrder| QuerySingleFieldOrder | QueryManyFieldsManySortingOrder | QueryManyFieldsSingleSortingOrder): String =
 orderBy match {
    case noOrder if noOrder is QueryFieldNoOrder or isEmpty(noOrder) -> ""
    case singleFieldOrder is QuerySingleFieldOrder -> " ORDER BY " ++ (singleFieldOrder.field ++ " " ++ singleFieldOrder.order)
    case manyFieldSingleOrder is QueryManyFieldsSingleSortingOrder -> " ORDER BY " ++ (manyFieldSingleOrder.fields joinBy  ", " ++ " " ++ manyFieldSingleOrder.order)
    case manyMany is QueryManyFieldsManySortingOrder -> " ORDER BY " ++ (manyMany.fields map getFieldOrder($) joinBy  ", ")
    else -> ""
}

fun queryToSoql(query: QueryBuilderInputType): String = "SELECT $(getFieldSelection(query.projectedFields)) FROM $(query.objectType)$(getFilter(query.filter))$(getOrderBy(query.orderBy))$(getLimit(query.limit))$(getOffset(query.offset))"

type QueryBuilderInputType = @QueryBuilder() {
    objectType: @ValuesFrom(value = { name: "valueProviderName" }) String,
    projectedFields: @FieldSelector()
                     @Discriminator(value = {key: "kind", defaultSelection: "ALL"})
                        (AllFieldsSelection | CustomFieldsSelection<@ValuesFrom(value = { name: "projectedFieldsValueProvider", arguments: {objectType: "/objectType"}}) String>),
    filter: @ResultSetFilter()
              @Discriminator(value = {key: "kind"}) QueryFilter<@ValuesFrom(value = { name: "filterFieldsValueProvider", arguments: {objectType: "/objectType"}}) String, @ValuesFrom(value = { name: "filterOperatorValueProvider" }) String>,
    limit: @FieldLimit() Number,
    offset: Number,
    orderBy: @FieldOrder() @MinSize(value = 1) @MaxSize(value = 10) QueryFieldNoOrder |
     QuerySingleFieldOrder<@ValuesFrom(value = { name: "orderFieldsValueProvider", arguments: {objectType: "/objectType"}}) String> |
     QueryManyFieldsManySortingOrder<@ValuesFrom(value = { name: "orderFieldsValueProvider", arguments: {objectType: "/objectType"}}) String> |
     QueryManyFieldsSingleSortingOrder<@ValuesFrom(value = { name: "orderFieldsValueProvider", arguments: {objectType: "/objectType"}}) String> 
}

type DynamicOutputType = @MetadataProvider(value = {name: "metadataProviderForProjection", arguments: {objectType: "/objectType", fields: "/projectedFields"}}) Object

var queryBuilder : Operation<QueryBuilderInputType, QueryResult<DynamicOutputType>, ResultFailure<GenericErrorResult, Error>, HttpConnection> = {
    name: "queryBuilder",
    displayName: "Query",
    executor: (parameter, connection) -> do {
        var query = queryToSoql(parameter)
        var response = {
                 contentType: "application/json",
                 status: 200,
                 headers: {},
                 cookies: {},
                 body: {
                    records: [
                      {
                        name: "query",
                        description: query
                      }
                    ]
                 }
        }
        ---
        if (response.status == 200)
          success(response.body as QueryResult<DynamicOutputType>)
        else
          failure(response.body as GenericErrorResult)
    }
}

var queryBuilderPaginated = offsetPaginated(
    queryBuilder,
    (page) -> page.records default [],
    (param) -> param.offset default 0,
    (param, offset) -> (param update { case off at .offset! -> offset })
)

var queryBuilderTriggerStrategy : TriggerStrategy<QueryResult<DynamicOutputType>, Object, Object, String> = {
    items: (result) -> result.records,
    item: (item) -> item,
    watermark: (result, item) -> item.id as String,
    identity: (item) -> item.id as String,
    watermarkCompareTo: DefaultWatermarkComparison
}

var queryBuilderTrigger = {
      name: "queryBuilderTrigger",
      displayName: "Query Builder Trigger",
      metadata: {
          order: "ASC",
          paginated: false
      },
      strategy: queryBuilderTriggerStrategy,
      operation: queryBuilder,
      inputMapper: (param: QueryBuilderInputType, watermark: String) -> param,
      initialWatermark: (triggerInput, connection) -> -1
}

type QueryResult<RecordType> = {
  records: Array<RecordType>
}

type GenericError = {
    errorCode?: String,
    message?: String,
    errorCode?: String,
    fields?: Array<String>,
    extendedErrorDetails?: Array<{ extendedErrorCode?: String }>
}

type GenericErrorResult = Array<GenericError>
