%dw 2.8

import * from dw::core::Arrays

import * from com::mulesoft::connectivity::Model
import * from com::mulesoft::connectivity::decorator::Annotations
import * from com::mulesoft::connectivity::decorator::Operation
import * from com::mulesoft::connectivity::decorator::Trigger
import * from com::mulesoft::connectivity::Metadata
import * from com::mulesoft::connectivity::transport::Http
import * from com::mulesoft::connectivity::decorator::ValueProvider
import * from com::mulesoft::connectivity::metadata::Builder
import * from com::mulesoft::connectivity::Types

// Type
type AccountUpdate = {name: String, email: String, createdAt?: LWDateTime, price?: Double, priority?: 1 | 2 | 3 | 4 | 5, role?: "admin" | "user"}
type Account = {id: String} & AccountUpdate

type AccountUpdateWrapper = {account : AccountUpdate}
// Connection

@OperationElement
var isAlive: Operation<{}, HttpResponse, ResultFailure<HttpResponse, Error>, HttpConnection> = {
    name: "isAlive",
    displayName: "Is alive",
    executor: (parameter, connection) -> success(
        	{
                contentType: "application/json",
                status: 200,
                headers: {},
                cookies: {}
    		})
}

@TestConnectionElement()
var testConnection = {
	validate: defineTestConnection(
        isAlive,
        (response) -> { isValid: response.value.status == 200 }
    )
}

@ConnectionElement()
var connectionProvider = defineBasicHttpConnectionProvider<BasicAuthSchema & { baseUri: String }>(
    (connectionInfo) -> {username: connectionInfo.username, password: connectionInfo.password},
    (parameter) -> {baseUri: parameter.baseUri}
)

@ConnectionElement()
var oauth2ClientCredentialsConnectionProvider = defineOAuth2Connection<{baseUri: String, accessToken: String}>(
    (parameter) -> {accessToken: parameter.accessToken},
    (parameter) -> {baseUri: parameter.baseUri},
    {
         grantType: 'clientCredentials',
         refreshUrl: "http://localhost/refresh",
         tokenUrl: "http://localhost/token",
         scopes: ["test"]
    }
)

@ConnectionElement()
var apiKeyHeaderConnectionProvider = defineApiKeyHttpConnectionProvider<{baseUri: String, apiKey: String}>(
    (parameter) -> {apiKey: parameter.apiKey},
    (parameter) -> {baseUri: parameter.baseUri},
    {in: 'header', name: "X-API-KEY"}
)

@ConnectionElement()
var oAuthAuthorizationCodeExtensions = defineOAuth2Connection<{baseUri: String, accessToken: String}>(
    (parameter) -> {accessToken: parameter.accessToken},
    (parameter) -> {baseUri: parameter.baseUri},
    {
        grantType: 'authorizationCode',
        authorizationUrl: "http://localhost/authorize",
        tokenUrl: "http://localhost/token",
        scopes: ["test"]
    },
   (parameter) ->   [
                        {inDance: true, in: "header", name:"tenantid", value: parameter.accessToken}
                    ]
)


type Error404 = Error<"404", "NOT_FOUND">

var error404 : Error404 = {
    kind: "404",
    categories : ["NOT_FOUND"]
}

type Error400 = Error<"400", "BAD_REQUEST">

var error400 : Error400 = {
    kind: "400",
    categories : ["BAD_REQUEST"]
}

// Operation
@OperationElement()
var updateAccountById : Operation<{accountId: @ValuesFrom(value = { name: "getAccountsValueProvider" }) String, account: AccountUpdate},
    HttpResponse<{id: String, name: String, email: String}>,
    ResultFailure<HttpResponse, Error404 | Error400>,
    HttpConnection> = {
    name: "updateAccountById",
    displayName: "Update Account by Id",
    executor: (parameter, connection) -> parameter.accountId match {
        case "999" -> do {
                var response = {
                                contentType: "application/json",
                                status: 404,
                                statusText: "Not Found",
                                headers: {},
                                cookies: {},
                                body: {} <~ {mimeType: "application/json; charset=utf-8", raw: "" as Binary {base: "64"}}
                        }
                ---
                failure(response, error404,response.body.detail)
            }
        case "-1" -> do {
                var response = {
                                contentType: "application/json",
                                status: 400,
                                headers: {},
                                cookies: {},
                                body: {
                                    error: "invalidAccountId",
                                    detail: "accountId cannot be a negative value"
                                } <~ {mimeType: "application/json; charset=utf-8", raw: "" as Binary {base: "64"}}
                        }
                ---
                failure(response, error400,response.body.detail)
            }
        else ->
            success({
                contentType: "application/json",
                status: 200,
                headers: {},
                cookies: {},
                body: {
                    id: parameter.accountId,
                    name: parameter.account.name,
                    email: parameter.account.email
                } <~ {mimeType: "application/json; charset=utf-8", raw: "" as Binary {base: "64"}}
            })
        }
}

@OperationElement()
var getAccounts : Operation<{offset: Number, limit: Number}, HttpResponse<{items: Array<{id: String, name: String, email: String}>}>, ResultFailure<HttpResponse, Error>, HttpConnection> = {
    name: "getAccounts",
    displayName: "Get Accounts",
    executor: (parameter, connection) -> success({
          contentType: "application/json",
          status: 200,
          headers: {},
          cookies: {},
          body: {
            items: slice([
              {
                id: "1",
                name: "One",
                email: "one@acme.com"
              },
              {
                id: "2",
                name: "Two",
                email: "two@acme.com"
              },
              {
                id: "3",
                name: "Three",
                email: "three@acme.com"
              },
              {
                id: "4",
                name: "Four",
                email: "four@acme.com"
              },
            ], parameter.offset, parameter.offset + parameter.limit)
          }
    })
}

var getAccountsPaginated: Operation<{offset: Number,limit: Number}, Page<Account,
  {offset: Number, limit: Number}>, ResultFailure<HttpResponse<Any>, Error>, HttpConnection> = getAccounts paginated (param, page) -> {
    items: page.body.items default [],
    (nextPage: {
        args: param update { case offset at .offset -> offset + sizeOf(page.body.items) }
    }) if !isEmpty(page.body.items)
}

var accountsValueProvider =
	defineValueProvider(getAccounts.executor, (response: HttpResponse<{items: Array<{id: String, name: String, email: String}>}>) -> response.body.items map ((item, index) -> {value: item.id, displayValue: { label: item.name!}}))


var orderType : TypeValueDefinition =
    createObjectTypeValueDefinition("Order")
      withField
        createField("orderId", createStringTypeValueDefinition(), true)
      withField
        createField("description",createStringTypeValueDefinition(), false)

type MetadataProviderInputType = {objectType: String}
var metadataProvider : MetadataProviderExecutor<MetadataProviderInputType, {errorMessage: String}, HttpConnection> =
    (key, conn) -> success(orderType)


type GetUsersRequest = {since: Number}
type GetUsersResponse = {items: Array<User>}
type User = {id: String, name: String, email: String, created_at: Number}

var getUsers : HttpOperation<GetUsersRequest, GetUsersResponse, HttpResponse, HttpConnection> = {
    name: "getUsers",
    displayName: "Get Users",
    executor: (parameter, connection) -> success({
          contentType: "application/json",
          status: 200,
          headers: {},
          cookies: {},
          body: {
                  items: [
                      {
                          id: 1,
                          name: "pepe",
                          email: "p@sales.com",
                          created_at: 20200102
                      },
                      {
                          id: 2,
                          name: "pepito",
                          email: "pe@sales.com",
                          created_at: 20200105
                      },
                      {
                          id: 3,
                          name: "pepillo",
                          email: "pep@sales.com",
                          created_at: 20210517
                      },
                    {
                        id: 4,
                        name: "pepinillo",
                        email: "pepi@sales.com",
                        created_at: 20210517
                    }
                  ]
          }
    } as HttpResponse<GetUsersResponse>)
}

var usersValueProvider =
	defineValueProvider(getUsers.executor, (response: HttpResponse<{items: Array<{id: String, name: String, email: String}>}>) -> response.body.items map ((item, index) -> {value: item.id, displayValue: { label: item.name, properties: {email: item.email}}}))

var getUserById : Operation<{userId: @ValuesFrom(value = { name: "usersValueProvider" }) String}, HttpResponse<{id: String, name: String, email: String}>, ResultFailure<HttpResponse, Error404 | Error400>, HttpConnection> = {
  name: "getUserByIdEndpoint",
  displayName: "Get user by id",
  executor: (parameter, connection) ->
    success({
      contentType: "application/json",
      status: 200,
      headers: {},
      cookies: {},
      body: {
        id: "1",
        name: "Name",
        email: "name@email.com"
      }
  })
}

type NewGetUsersRequest = GetUsersRequest & {newRequiredParam: Number}
var newGetUsers =
    getUsers mapInputOperation (i:NewGetUsersRequest) -> i ++ {newRequiredParam: 0}

var myTriggerStrategy:TriggerStrategy<HttpResponse<GetUsersResponse>,User,User,Number> =
{
    items: (result) -> result.body.items,
    item: (item) -> item,
    watermark: (result,item) -> item.created_at,
    identity: (item) -> item.id as String,
    watermarkCompareTo: DefaultWatermarkComparison
}

@TriggerElement()
var getUsersTrigger = {
    name: "getUsersTrigger",
    displayName: "Get Users Trigger",
    metadata: {
        order: "ASC",
        paginated: false
    },
    strategy: myTriggerStrategy,
    operation: getUsers as Operation<GetUsersRequest, HttpResponse<GetUsersResponse>, ResultFailure<HttpResponse, Error>, HttpConnection>,
    inputMapper: (ti: GetUsersRequest, w) -> ti,
    initialWatermark: (triggerInput: GetUsersRequest, connection) -> 20200101,
    versions: {
        "2": {
            metadata: {
                order: "ASC",
                paginated: false
            },
            strategy: myTriggerStrategy,
            operation: newGetUsers as Operation<NewGetUsersRequest, HttpResponse<GetUsersResponse>, ResultFailure<HttpResponse, Error>, HttpConnection>,
            inputMapper: (ti: NewGetUsersRequest, w) -> ti,
            initialWatermark: (NewGetUsersRequest, HttpConnection) -> 20200101, //(now() as String { format: "uuuuMMdd" }) as Number
        }
    }
}
