%dw 2.8

import * from com::mulesoft::connectivity::Model
import * from com::mulesoft::connectivity::decorator::Annotations
import * from com::mulesoft::connectivity::decorator::Operation
import * from com::mulesoft::connectivity::decorator::Trigger
import * from com::mulesoft::connectivity::Metadata
import * from com::mulesoft::connectivity::transport::Http
import * from com::mulesoft::connectivity::decorator::ValueProvider
import * from com::mulesoft::connectivity::decorator::Executor

// ----------------
// ---- Types -----
// ----------------
type Account = {
    sid?: String, //{"description": "A 34 character string that uniquely identifies this resource."},
    date_created?: String, //{"description": "The date that this account was created, in GMT in RFC 2822 format"},
    date_updated?: String, //{"description": "The date that this account was last updated, in GMT in RFC 2822 format."},
    friendly_name?: String, //{"description": "A human readable description of this account, up to 64 characters long. By default the FriendlyName is your email address."},
    owner_account_sid?: String, //{"description": "The unique 34 character id that represents the parent of this account. The OwnerAccountSid of a parent account is its own sid."},
    status?: String, //{"description": "The status of this account. Usually `active`, but can be `suspended` or `closed`."},
    "type"?: String, //{"description": "The type of this account. Either `Trial` or `Full` if it's been upgraded"},
    auth_token?: String, //{"description": "The authorization token for this account. This token should be kept a secret, so no sharing"},
    subresource_uris?: {}, //{"description": "A Map of various subresources available for the given Account Instance"},
    uri?: String, //{"description": "The URI for this resource, relative to `https://api.twilio.com`"}
}

type Message = {
    sid?: String, //{"description": "The unique string that that we created to identify the Message resource."},
    body?: String, //{"description": "The message text. Can be up to 1,600 characters long"},
    num_segments?: String, //{"description": "The number of segments that make up the complete message. A message body that is too large to be sent in a single SMS message is segmented and charged as multiple messages. Inbound messages over 160 characters are reassembled when the message is received. Note: When using a Messaging Service to send messages, `num_segments` will always be 0 in Twilio''s response to your API request."},
    direction?: String, //{"description": "The direction of the message. Can be: `inbound` for incoming messages, `outbound-api` for messages initiated by a REST API, `outbound-call` for messages initiated during a call, or `outbound-reply` for messages initiated in response to an incoming message."},
    from?: String, //{"description": "The phone number (in [E.164](https://en.wikipedia.org/wiki/E.164) format), [alphanumeric sender ID](https://www.twilio.com/docs/sms/send-messages#use-an-alphanumeric-sender-id), or [Wireless SIM](https://www.twilio.com/docs/wireless/tutorials/communications-guides/how-to-send-and-receive-text-messages) that initiated the message. For incoming messages, this will be the number of the sending phone. For outgoing messages, this value will be one of your Twilio phone numbers or the alphanumeric sender ID used."},
    to?: String, //{"description": "The phone number in [E.164](https://en.wikipedia.org/wiki/E.164) format that received the message. For incoming messages, this will be one of your Twilio phone numbers. For outgoing messages, this will be the sending phone."},
    date_created?: String, //{"description": "The date and time in GMT that the resource was created specified in [RFC 2822](https://www.ietf.org/rfc/rfc2822.txt) format."},
    date_updated?: String, //{"description": "The date and time in GMT that the resource was last updated specified in [RFC 2822](https://www.ietf.org/rfc/rfc2822.txt) format."},
    date_sent?: String, //{"description": "The date and time in GMT that the resource was sent specified in [RFC 2822](https://www.ietf.org/rfc/rfc2822.txt) format. For outgoing messages, this is when we sent the message. For incoming messages, this is when we made the HTTP request to your application."},
    price?: String, //{"description": "The amount billed for the message, in the currency specified by `price_unit`.  Note that your account is charged for each segment we send to the handset. Populated after the message has been sent. May not be immediately available."},
    error_message?: String, //{"description": "The description of the `error_code` if your message `status` is `failed` or `undelivered`. If the message was successful, this value is null."},
    account_sid?: String, //{"description": "The SID of the [Account](https://www.twilio.com/docs/iam/api/account) that sent the message that created the resource."},
    num_media?: String, //{"description": "The number of media files associated with the message. A message can send up to 10 media files."},
    status?: String, //{"description": "The status of the message. Can be: `accepted`, `scheduled`, `canceled`, `queued`, `sending`, `sent`, `failed`, `delivered`, `undelivered`, `receiving`, `received`, or `read` (WhatsApp only). For more information, See [detailed descriptions](https://www.twilio.com/docs/sms/api/message-resource#message-status-values)."},
    messaging_service_sid?: String, //{"description": "The SID of the [Messaging Service](https://www.twilio.com/docs/sms/services/api) used with the message. The value is null if a Messaging Service was not used."},
    error_code?: Number, //{"description": "The error code returned if your message `status` is `failed` or `undelivered`. The error_code provides more information about the failure. If the message was successful, this value is null."},
    price_unit?: String, //{"description": "The currency in which `price` is measured, in [ISO 4127](https://www.iso.org/iso/home/standards/currency_codes.htm) format (e.g. `usd`, `eur`, `jpy`)."},
    api_version?: String, //{"description": "The API version used to process the message."},
    uri?: String, //{"description": "The URI of the resource, relative to `https://api.twilio.com`."},
    subresource_uris?: {}, //{"description": "A list of related resources identified by their URIs relative to `https://api.twilio.com`"}
}

// ----------------
// -- Connection --
// ----------------
type BasicConnection = @Label(value = "Twilio basic connection") {
  baseUri: @Label(value = "The base uri that will be used for all HTTP requests") String,
  username: @Label(value = "The username used to authenticate the requests") String,
  password: @Label(value = "The password used to authenticate the requests") @SemanticTerms(value = ["password"]) String
}

@ConnectionElement()
var basicAuthConnection: BasicHttpConnectionProvider<BasicConnection> = defineBasicHttpConnectionProvider<BasicConnection>(
	(parameter: BasicConnection) -> {
         username: parameter.username,
         password: parameter.password
	 },
	 (parameter: BasicConnection) -> { baseUri: parameter.baseUri }
)

@ConnectionElement()
var oauth2AuthCodeConnectionProvider = defineOAuth2Connection<{baseUri: String, accessToken: String}>(
    (parameter) -> {accessToken: parameter.accessToken},
    (parameter) -> {baseUri: parameter.baseUri},
    {
        grantType: 'authorizationCode',
        authorizationUrl: "http://localhost/authorize",
        tokenUrl: "http://localhost/token",
        scopes: ["test"]
    }
)

@ConnectionElement()
var apiKeyHeaderConnectionProvider = defineApiKeyHttpConnectionProvider<{baseUri: String, apiKey: String}>(
    (parameter) -> {apiKey: parameter.apiKey},
    (parameter) -> {baseUri: parameter.baseUri},
    {in: 'header', name: "X-API-KEY"}
)

// ----------------
// -- Operations --
// ----------------
type TwilioPage = {
  page: Number,
  page_size: Number,
  uri: String,
  first_page_uri: String,
  previous_page_uri: String | Null,
  next_page_uri: String | Null,
}

type AccountsPage = TwilioPage & {
	accounts: Array<Account>
}

type MessagesPage = TwilioPage & {
	messages: Array<Message>
}

type PaginationParam = {
  page?: String,
  pageSize?: String,
  afterSid?: String
}

type AccountsRequestType = HttpRequestType<{|
	query: PaginationParam,
	headers: {},
	cookie: {}
|}>

type MessagesRequestType = HttpRequestType<{|
    uri: {
        accountId: @ValuesFrom(value = { name: "accountsValueProvider" }) String
    },
    query: PaginationParam,
    headers: {},
    cookie: {}
|}>

fun getExecutor<OpParam <: BaseHttpRequestType, ResultTwilioType>(httpMethod: HttpMethod, resourceBuilder: (OpParam) -> String):
	(OpParam, HttpConnection) -> Result<HttpResponse<ResultTwilioType>, ResultFailure<HttpResponse, Error>> =
	(parameter, connection) -> do {
        var response = connection({
        	method: httpMethod,
            path: resourceBuilder(parameter),
            (queryParams: parameter.query) if (parameter != null)
        })
        ---
        if (response.status == 200)
        	success(response as HttpResponse<ResultTwilioType>)
        else
            failure(response)
    }

var getAccountsOperation: HttpOperation<AccountsRequestType, AccountsPage, Any, HttpConnection> = {
    name: "getAccounts",
    displayName: "Get Accounts",
    executor: getExecutor<AccountsRequestType, AccountsPage>("GET",(param: AccountsRequestType) -> "/2010-04-01/Accounts.json")
}



@OperationElement()
var getAccounts: Operation<AccountsRequestType, Page<Account, AccountsRequestType>, ResultFailure<HttpResponse, Error>, HttpConnection> =
    getAccountsOperation paginated (param, pageValue) -> {
    items: pageValue.body.accounts default [],
    (nextPage: {
        args: param  update {
        	case .query! -> parseQueryParameters(pageValue.body.next_page_uri)
        }
    }) if (!isEmpty(pageValue.body.accounts) and !isEmpty(pageValue.body.next_page_uri))
}

var getMessagesOperation: HttpOperation<MessagesRequestType, MessagesPage, Any, HttpConnection> = {
    name: "getMessages",
    displayName: "Get Messages",
    executor: getExecutor<MessagesRequestType,MessagesPage>("GET",(param: MessagesRequestType) -> "/2010-04-01/Accounts/" ++ param.uri.accountId ++ "/Messages.json")
}

@OperationElement()
var getMessages: Operation<MessagesRequestType, Page<Message, MessagesRequestType>, ResultFailure<HttpResponse, Error>, HttpConnection> =
    getMessagesOperation paginated (param, pageValue) -> {
    items: pageValue.body.messages default [],
    (nextPage: {
        args: param  update {
        	case .query! -> parseQueryParameters(pageValue.body.next_page_uri)
        }
    }) if (!isEmpty(pageValue.body.messages) and !isEmpty(pageValue.body.next_page_uri))
}

// ---------------------
// -- Test Connection --
// ---------------------
@TestConnectionElement()
var testConnection: HttpTestConnection = {
	validate: defineTestConnection(
              	getAccountsOperation mapInputOperation (parameter: {||}) -> {query: { pageSize: "1" }, headers: {}, cookie: {}},
              	(response) -> { 	isValid: response.value.status == 200,
                                      message: if (response.success) "Connection test succeeded" else "Connection test failed",
                                      (error: response.description default "") if (!isEmpty(response.description))
                                  }
              )
}

var emptyAccountRequest: AccountsRequestType = {
    query: {},
    headers: {},
    cookie: {}
}

var accountsValueProviderExecutor: Executor<{}, Array<Account>, ResultFailure<HttpResponse, Error>, HttpConnection> =
    mapSuccessResult(mapInput (getAccounts.executor, (newInput: {}) -> emptyAccountRequest),
(response: Page<Account, AccountsRequestType>) -> response.items)

var accountsValueProvider: ValueProvider<{}, HttpResponse<Any>, String | Null, Object, HttpConnection> = defineValueProvider(accountsValueProviderExecutor, (response: Array<Account>) -> response map ((item, index) -> {value: item.sid, displayValue: { label: item.friendly_name!}}))

type GetUsersRequest = {since: Number}
type GetUsersResponse = {items: Array<User>}
type User = {id: String, name: String, email: String, created_at: Number}

var getUsers : HttpOperation<GetUsersRequest, GetUsersResponse, HttpResponse, HttpConnection> = {
    name: "getUsers",
    displayName: "Get Users",
    executor: (parameter, connection) -> success({
          contentType: "application/json",
          status: 200,
          headers: {},
          cookies: {},
          body: {
                  items: [
                      {
                          id: 1,
                          name: "pepe",
                          email: "p@sales.com",
                          created_at: 20200102
                      },
                      {
                          id: 2,
                          name: "pepito",
                          email: "pe@sales.com",
                          created_at: 20200105
                      },
                      {
                          id: 3,
                          name: "pepillo",
                          email: "pep@sales.com",
                          created_at: 20210517
                      },
                    {
                        id: 4,
                        name: "pepinillo",
                        email: "pepi@sales.com",
                        created_at: 20210517
                    }
                  ]
          }
    } as HttpResponse<GetUsersResponse>)
}

var myTriggerStrategy:TriggerStrategy<HttpResponse<GetUsersResponse>,User,User,Number> =
{
    items: (result) -> result.body.items,
    item: (item) -> item,
    watermark: (result,item) -> item.created_at,
    identity: (item) -> item.id as String,
    watermarkCompareTo: DefaultWatermarkComparison
}

@TriggerElement()
var getUsersTrigger: Trigger<GetUsersRequest, GetUsersRequest, HttpResponse<GetUsersResponse>, GetUsersRequest, ResultFailure<HttpResponse, Error>, HttpConnection, User, User, Number> = {
    name: "getUsersTrigger",
    displayName: "Get Users Trigger",
    metadata: {
        order: "ASC",
        paginated: false
    },
    strategy: myTriggerStrategy,
    operation: getUsers,
    inputMapper: (ti, w) -> ti,
    initialWatermark: (triggerInput, connection) -> 20200101
}

type GetUsersRequestPaginated = {since: Number, offset?: Number, limit: Number }

var getUsersToBePaginated : Operation<GetUsersRequestPaginated, HttpResponse<GetUsersResponse>, ResultFailure<HttpResponse, Error>, HttpConnection> = {
    name: "getUsers",
    displayName: "Get Users",
    executor: (parameter, connection) -> do {
 if (parameter.offset? == false)
        success({
                          contentType: "application/json",
                          status: 200,
                          headers: {},
                          cookies: {},
                          body: {
                    items: [
                      {
                          id: 1,
                          name: "pepe",
                          email: "p@sales.com",
                          created_at: 20200102
                      },
                      {
                          id: 2,
                          name: "pepito",
                          email: "pe@sales.com",
                          created_at: 20200105
                      },
                      {
                          id: 3,
                          name: "pepillo",
                          email: "pep@sales.com",
                          created_at: 20210517
                      }
                    ]
                }
                 } as HttpResponse<GetUsersResponse>)
         else
                 success({
                                   contentType: "application/json",
                                   status: 200,
                                   headers: {},
                                   cookies: {},
                                   body: {
                             items: [
                               {
                                   id: 4,
                                   name: "juan",
                                   email: "j@sales.com",
                                   created_at: 20210519
                               },
                               {
                                   id: 5,
                                   name: "perez",
                                   email: "perez@sales.com",
                                   created_at: 20211111
                               },
                               {
                                   id: 6,
                                   name: "jose",
                                   email: "jose@sales.com",
                                   created_at: 20230609
                               }
                             ]
                         }
                          } as HttpResponse<GetUsersResponse>)
    }
}

var getUsersPaginated = (getUsersToBePaginated paginated (param, page) -> {
    items: page.body.items default [],
    (nextPage: {
        args: param update { case offset at .offset! -> offset default 0 + sizeOf(page.body.items) }
    }) if !isEmpty(page.body.items)
})

var myTriggerStrategyPaginated:TriggerStrategy<Page<User, GetUsersRequestPaginated>,User,User,Number>  =
{
    items: (result) -> result.items,
    item: (item) -> item,
    watermark: (result,item) -> item.created_at,
    identity: (item) -> item.id as String,
    watermarkCompareTo: DefaultWatermarkComparison
}

@TriggerElement()
var getUsersTriggerPaginated: Trigger<GetUsersRequestPaginated, GetUsersRequestPaginated, Page<User, GetUsersRequestPaginated> ,
GetUsersRequestPaginated, ResultFailure<HttpResponse, Error>, HttpConnection, User, User, Number> = {
    name: "getUsersTriggerPaginated",
    displayName: "Get Users Trigger",
    metadata: {
        order: "ASC",
        paginated: true
    },
    strategy: myTriggerStrategyPaginated,
    operation: getUsersPaginated,
    inputMapper: (ti, w) -> ti,
    initialWatermark: (triggerInput, connection) -> 20200101
}

