/*
 * Copyright (c) 2025, Salesforce, Inc.,
 * All rights reserved.
 * For full license text, see the LICENSE.txt file
 */
%dw 2.8

import * from com::mulesoft::connectivity::Types
import * from com::mulesoft::connectivity::decorator::Annotations

/**
 * Type that represents a unary query condition (i.e.: "name isNull").
 */
type UnaryQueryCondition<F <: String, O <: String> = {
    field: @RequiresValueProvider(value = {providedValueType: String, displayPropertiesType: FieldType}) F,
    operator: @RequiresValueProvider(value = {providedValueType: String, displayPropertiesType: OperatorOperandTypes}) O
}

/**
 * Type that represents a binary query condition (i.e.: "name = 'John'").
 */
type BinaryQueryCondition<F <: String, O <: String> = {
    field: @RequiresValueProvider(value = {providedValueType: String, displayPropertiesType: FieldType}) F,
    operator: @RequiresValueProvider(value = {providedValueType: String, displayPropertiesType: OperatorOperandTypes}) O,
    value: Any
}

/**
 * Class that represents a N-ary query condition (i.e.: "age between 18 AND 60").
 */
type NAryQueryCondition<F <: String, O <: String> = {
    field: @RequiresValueProvider(value = {providedValueType: String, displayPropertiesType: FieldType}) F,
    operator: @RequiresValueProvider(value = {providedValueType: String, displayPropertiesType: OperatorOperandTypes}) O,
    values: Array<Any>
}

/**
* Union of all types of query conditions.
*/
type QueryCondition<F <: String, O <: String> = UnaryQueryCondition<F, O> | BinaryQueryCondition<F, O> | NAryQueryCondition<F, O>

/**
 * Type that represents there are no filters in a query.
 */
type NoQueryFilter = {
    kind: @Label(value = "None - Get All Records") "NONE"
}

/**
 * Type that represents the filters of a query with 'AND' logical operator.
 */
type AndQueryFilter<F <: String, O <: String> = {
    kind: @Label(value = "All Conditions Are Met (AND)") "AND",
    conditions: Array<QueryCondition<F, O>>,
}

/**
 * Type that represents the filters of a query with 'OR' logical operator.
 */
type OrQueryFilter<F <: String, O <: String> = {
    kind: @Label(value = "Any Condition Is Met (OR)") "OR",
    conditions: Array<QueryCondition<F, O>>
}

/**
 * Type that represents the filters of a query with a custom expression.
 */
type CustomQueryFilter<F <: String, O <: String> = {
    kind: @Label(value = "Custom Condition Logic Is Met") "CUSTOM",
    expression: String,
    conditions: Array<QueryCondition<F, O>>
}

type QueryFilter<F <: String, O <: String> = NoQueryFilter | AndQueryFilter<F, O> | OrQueryFilter<F, O> | CustomQueryFilter<F, O>

/**
 * Type that represents select all projection fields.
 */
type AllFieldsSelection = {
  kind: @Label(value = "Select All Fields") "ALL"
}

/**
 * Type that represents select specific projection fields.
 */
type CustomFieldsSelection<F <: String> = {
  kind: @Label(value = "Select Custom Fields") "CUSTOM",
  fields: Array<@RequiresValueProvider(value = {providedValueType: String, displayPropertiesType: FieldType}) F>
}

type QueryOrder = "ASC" | "DESC"

/**
 * Base type that represents the order part of a query.
 */
type QueryFieldOrder = {|
    kind: "SORTED"
|}

/**
 * Type that represents the order part of a query and determinate a not sorted type
 */
type QueryFieldNoOrder = {|
    kind: "NOT_SORTED"
|}

/**
 * Type that represents a pair of field and order for a query.
 */
type QueryFieldOrderPair<F <: String> = {|
   field: @RequiresValueProvider(value = {providedValueType: String, displayPropertiesType: FieldType}) F,
   order: QueryOrder
|}

/**
 * Type that represents the order part of a query (i.e.: ORDER BY id asc).
 */
type QuerySingleFieldOrder<F <: String> = QueryFieldOrder & QueryFieldOrderPair<F>

/**
 * Type that represents the order part of a query (i.e.: ORDER BY id,name asc).
 */
type QueryManyFieldsSingleSortingOrder<F <: String> = QueryFieldOrder & {|
    fields: Array<QueryFieldOrderPair.field>,
    order: QueryFieldOrderPair.order
|}

/**
 * Type that represents the order part of a query (i.e.: ORDER BY id asc, name desc).
 */
type QueryManyFieldsManySortingOrder<F <: String> = QueryFieldOrder & {|
    fields: Array<QueryFieldOrderPair<F>>
|}

type OperatorOperandTypes = {
   operandTypes: Array<Array<OperandType>>
}

type FieldType = {
   fieldType: OperandType
}

/**
* Possible operand types.
*/
type OperandType = Type<String> | Type<Number> | Type<Boolean> | Type<LWDate> | Type<LWDateTime> | Type<StringArray> | Type<NumberArray> | Type<Null> | Type<Date> | Type<DateTime> | Type<BooleanArray> | Type<DateArray> | Type<DateTimeArray>
