/*
 * Copyright (c) 2025, Salesforce, Inc.,
 * All rights reserved.
 * For full license text, see the LICENSE.txt file
 */
/**
 * Includes useful annotations that allow connector developers to add labels,
 * descriptions, semantic terms, value and metadata providers to their types.
 */
%dw 2.8

/**
 * Describes the expected values for a given field.
 */
@AnnotationTarget(targets = ["KeyType", "Type", "TypeExpression"])
@Metadata(key = "description")
annotation Description(value: String)

/**
 * The label to show for a given field.
 */
@AnnotationTarget(targets = ["KeyType", "Type", "TypeExpression"])
@Metadata(key = "label")
annotation Label(value: String)

/**
 * The help text to show for a given field. It differs from the description in that this indicates the way the field should be used.
 */
@AnnotationTarget(targets = ["KeyType", "Type", "TypeExpression"])
@Metadata(key = "helpText")
annotation HelpText(value: String)

/**
 * Describes the role a given value has, allowing to classify it as sensitive
 * if necessary.
 */
@AnnotationTarget(targets = ["KeyType", "Type", "TypeExpression"])
@Metadata(key = "semanticTerms")
annotation SemanticTerms(value: Array<SemanticTerm>)

/**
 * Possible roles for a given value.
 */
type SemanticTerm = "username" | "password" | "secretToken" | "apiKey"

/**
 * Annotates a given field as having a value provider that can be used to
 * suggest valid invocation values.
 */
@AnnotationTarget(targets = ["KeyType", "Type", "TypeExpression"])
@Metadata(key = "valueProvider")
annotation ValuesFrom(value: ProviderDescription)

/**
 * Annotates a given field as having a metadata provider that can be used to
 * resolve the dynamic schema constraints of its values.
 */
@AnnotationTarget(targets = ["KeyType", "Type", "TypeExpression"])
@Metadata(key = "metadataProvider")
annotation MetadataProvider(value: ProviderDescription)

/**
 * Exposes some inner fields of the current type as contextual variables
 * available as arguments to both Value Providers and Metadata Providers.
 */
@AnnotationTarget(targets = ["KeyType", "Type", "TypeExpression"])
@Metadata(key = "with")
annotation With(value: ContextBinding)

/**
* Defines the discriminator in an Union
*/
@AnnotationTarget(targets = ["TypeExpression"])
@Metadata(key = "discriminator")
annotation Discriminator(value: DiscriminatorDescription)

/**
* Indicates the field will have to define a Value Provider.
*/
@AnnotationTarget(targets = ["KeyType", "Type", "TypeExpression"])
@Metadata(key = "requiresValueProvider")
annotation RequiresValueProvider(value: RequiresValueProviderDefinition)

/**
* Indicates the minimun size of the collection field.
*/
@AnnotationTarget(targets = ["KeyType", "Type", "TypeExpression"])
@Metadata(key = "minSize")
annotation MinSize(value: Number)

/**
* Indicates the maximum size of the collection field.
*/
@AnnotationTarget(targets = ["KeyType", "Type", "TypeExpression"])
@Metadata(key = "maxSize")
annotation MaxSize(value: Number)

/**
 * Default value for a given field.
 */
@AnnotationTarget(targets = ["KeyType", "Type", "TypeExpression"])
@Metadata(key = "defaultValue")
annotation DefaultValue(value: String)

/**
 * Hidden field
 */
@AnnotationTarget(targets = ["KeyType", "Type", "TypeExpression"])
@Metadata(key = "hidden")
annotation Hidden(value: {} = {})

/**
 * Describes a given Value Provider or Metadata Provider.
 *
 * Fields:
 * - `name`: The name of the provider to be invoked
 * - `arguments`: An object with a bunch of `parameterName -> binding` entries, for
 *   more information see `ContextBinding`
 */
type ProviderDescription = {|
  name: String,
  arguments?: ContextBinding
|}

type DiscriminatorDescription = {
  key: String,
  defaultSelection?: String
}

/**
 * A collection of `variableName -> binding` entries.
 *
 * Bindings have one of three shapes:
 * - `/`-prefixed: These bindings have their values fetched from the start of
 *   the parameter resolution scope. These bindings are not reusable across
 *   multiple operations but may be easier to code.
 * - `#`-prefixed: These bindings have their values fetched from the nearest
 *   contextual binding, see `@With` for more info.
 * - Plain selectors: These bindings select sibling fields from the current
 *   context.
 */
type ContextBinding = Dictionary<String>

/**
* Describes the types of the Value Provider to be defined.
* Fields:
* - `providedValueType`: The ProvidedValueType of the Value Provider.
* - `displayPropertiesType`: The DisplayPropertiesType of the Value Provider.
*/
type RequiresValueProviderDefinition = {
   providedValueType: Type,
   displayPropertiesType?: Type
}
