/*
 * Copyright (c) 2025, Salesforce, Inc.,
 * All rights reserved.
 * For full license text, see the LICENSE.txt file
 */
%dw 2.7
import * from com::mulesoft::connectivity::Model
import * from com::mulesoft::connectivity::decorator::Operation

/**
* Default watermarking comparison, for primitive types which support > , < comparison operator
*
* The method returns 0 if the previous value is equal to the new value. A value less than 0 is returned if the previous value is greater
* than the new value and a value greater than 0 if the new value is greater than the previous value.
*/
var DefaultWatermarkComparison = (prev,new) -> if(prev==new) 0 else if (new>prev) 1 else -1

/**
* Current response to a trigger output page type
* This function filters items according to the watermark and calculates the partial greatest watermark
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `watermarkCompareTo` | (Watermark,Watermark) -> Number | The watermark comparison function
* | `latestWatermark` | Watermark | The latest watermark, it means, watermark from previous poll (note, not previous page in same poll context)
* | `partialGreatestWatermark` | Watermark | The partial greatest watermark, only in pagination context, it means, current greatest watermark gotten in the already processed pages
* | `operationResult` | Page<TriggerItem<TriggerResulItemType,Watermark>,OperationInputType> | The gotten page which items are TriggerItem type
* |===
*
*/
@Internal(permits=[])
fun toTriggerPage<OperationInputType,TriggerResultItemType, Watermark>(
    watermarkCompareTo: (Watermark,Watermark) -> Number,
    latestWatermark: Watermark,
    partialGreatestWatermark: Null | Watermark, //Null when is no paginated or paginated first page
    operationResult: Page<TriggerItem<TriggerResultItemType,Watermark>,OperationInputType>
    ):TriggerPage<OperationInputType, TriggerResultItemType,Watermark> =
do
{
    // Filter items according to watermark
    // Weird case, offset pagination could loss items. If it calculates the offset according to the returned items and here those are filtered
    // because of watermark
    var validItems = (if(latestWatermark != null) operationResult.items filter (value, index) ->
        (watermarkCompareTo(latestWatermark,value.watermark) >= 0) else operationResult.items) default []

    // Get greatest watermark in current items
    var greatestWatermark: Watermark | Null = validItems.watermark reduce (
            if (watermarkCompareTo($$, $) >= 0) $
            else $$
        )

    var result =  operationResult update {
        case items at .items -> validItems
    }
    var greatestWatermarkObject: Watermark | Null = if(partialGreatestWatermark != null and greatestWatermark != null)
                                if(watermarkCompareTo(partialGreatestWatermark, greatestWatermark) >= 0)
                                    greatestWatermark
                                 else
                                    partialGreatestWatermark
                             else
                                partialGreatestWatermark default greatestWatermark

    var nextPoll: NextPoll<Watermark> = {latestWatermark: latestWatermark, greatestWatermark: greatestWatermarkObject default latestWatermark}
    ---
    result ++ {nextPoll: nextPoll}
}


/**
* No paginated response to trigger response
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `response` | OperationOutputType | Operation execution response
* | `triggerStrategy` | TriggerStrategy<OperationOutputType,TriggerResulItemType,OperationResultItemType,Watermark> | Trigger extraction strategy
* |===
*
*/
@Internal(permits=[])
fun nonPaginatedResponseToPageTriggerItemsTransformed<OperationInputType, OperationOutputType, TriggerResultItemType, OperationResultItemType, Watermark>(
    response: OperationOutputType,
    triggerStrategy: TriggerStrategy<OperationOutputType, TriggerResultItemType, OperationResultItemType, Watermark>
): Page<TriggerItem<TriggerResultItemType, Watermark>, OperationInputType> = do {
    // Transformation from operation output to trigger item's array
    var items = triggerStrategy.items(response)
        map ((item) -> {
            value: triggerStrategy.item(item),
            watermark: triggerStrategy.watermark(response, item),
            identity: triggerStrategy.identity(item)
        })
        default []
    ---
    {
        items: items
    }
}

/**
* Paginated response to trigger response
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `response` | Page<OperationResultItemType,OperationInputType> | Operation execution response
* | `triggerStrategy` | TriggerStrategy<OperationOutputType,TriggerResulItemType,OperationResultItemType,Watermark> | Trigger extraction strategy
* |===
*
*/
@Internal(permits=[])
fun paginatedResponseToPageTriggerItemsTransformed<OperationInputType, OperationOutputType, TriggerResultItemType, OperationResultItemType, Watermark>(
    response: Page<OperationResultItemType, OperationInputType>,
    triggerStrategy: TriggerStrategy<OperationOutputType, TriggerResultItemType, OperationResultItemType, Watermark>
): Page<TriggerItem<TriggerResultItemType, Watermark>, OperationInputType> = do {
    // Transformation from operation output to trigger item's array
    fun paginatedTriggerItemsTransformation(result: Array<OperationResultItemType>): Array<TriggerItem<TriggerResultItemType,Watermark>> =
        result
        map ((item, index) -> {
            value: triggerStrategy.item(item),
            watermark: triggerStrategy.watermark(result, item),
            identity: triggerStrategy.identity(item)
        })
    ---
    response update {
        case items at .items -> paginatedTriggerItemsTransformation(items)
    }
}

/**
* Executes a trigger poll
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `trigger` | Trigger<TriggerInputType, OperationInputType, OperationOutputType, OperationNextPageInputType,ResultErrorType, ConnectionType, TriggerResultItemType, OperationResultItemType, Watermark> | The trigger to execute the poll
* | `latestWatermark` | Watermark | Null | Latest watermark processed
* | `params` | TriggerInputType | OperationInputType | The parameters for the invocation
* | `connection` | ConnectionType | The connection used for the invocation
* |===
*
*/
fun executePolling<TriggerInputType, OperationInputType, OperationOutputType, ResultErrorType <: ResultFailure, TriggerResultItemType ,OperationResultItemType, OperationNextPageInputType, Watermark, ConnectionType>(
    trigger: Trigger<TriggerInputType, OperationInputType, OperationOutputType, OperationNextPageInputType, ResultErrorType, ConnectionType, TriggerResultItemType, OperationResultItemType, Watermark>,
    params: TriggerInputType,
    latestWatermark: Watermark,
    connection: ConnectionType
):

Result<TriggerPage<OperationNextPageInputType, TriggerResultItemType,Watermark> | TriggerPage<OperationInputType, TriggerResultItemType,Watermark>, ResultErrorType> =
    do {
      //Operation which the trigger is based on is executed
      var operationResult: Result<OperationOutputType, ResultErrorType> =
        trigger.operation.executor(trigger.inputMapper(params, latestWatermark), connection)
      ---
       operationResult match {
            case ok is ResultSuccess<OperationOutputType> ->
                ok.value match {
                case operationResultSuccess is Page<OperationResultItemType, OperationNextPageInputType> ->
                    //Paginated response
                     success(toTriggerPage<OperationNextPageInputType,TriggerResultItemType, Watermark>
                     (trigger.strategy.watermarkCompareTo , latestWatermark, null,
                      paginatedResponseToPageTriggerItemsTransformed<OperationNextPageInputType, OperationOutputType,TriggerResultItemType, OperationResultItemType, Watermark>
                      (operationResultSuccess, trigger.strategy)))

                case operationResultSuccess is OperationOutputType ->
                     // No paginated response
                     success(toTriggerPage<OperationInputType,TriggerResultItemType, Watermark>
                     (trigger.strategy.watermarkCompareTo , latestWatermark, null,
                     nonPaginatedResponseToPageTriggerItemsTransformed<OperationInputType, OperationOutputType,TriggerResultItemType, OperationResultItemType, Watermark>
                     (operationResultSuccess, trigger.strategy)))
            }
            case failure is ResultErrorType -> failure
        }
    }

/**
* Executes a trigger next page, same polling context
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `trigger` | Trigger<Any, OperationInputType, OperationOutputType, OperationNextPageInputType, ResultErrorType, ConnectionType, TriggerResultItemType ,OperationResultItemType, Watermark> | The trigger to execute the poll
* | `params` | TriggerInput<OperationInputType, Watermark> | The parameters for the invocation
* | `operationNextPage` | Operation<OperationNextPageInputType, Page<OperationNextPageOutputType, OperationResultItemType, OperationNextPageInputType, ConnectionType>, ResultErrorType, ConnectionType> | The operation to invoke to get next page
* | `connection` | ConnectionType | The connection used for the invocation
* |===
*
*/
fun executePollingNextPage<TriggerInputType, OperationInputType, OperationOutputType, ResultErrorType <: ResultFailure, TriggerResultItemType ,OperationResultItemType, Watermark, OperationNextPageInputType, ConnectionType>(
    trigger: Trigger<TriggerInputType, OperationInputType, OperationOutputType, OperationNextPageInputType, ResultErrorType, ConnectionType, TriggerResultItemType ,OperationResultItemType, Watermark>,
    params: OperationNextPageInputType,
    nextPoll: NextPoll<Watermark>,
    connection: ConnectionType
): Result<TriggerPage<OperationNextPageInputType, TriggerResultItemType,Watermark>, ResultErrorType> = do {
    // Operation which the trigger is based on is executed
    var operationResult = trigger.operation.nextPageExecutor(params, connection)
    ---
    operationResult match {
         case ok is ResultSuccess<Page<OperationResultItemType, OperationNextPageInputType>> ->
             // Paginated response
             success(toTriggerPage<OperationNextPageInputType,TriggerResultItemType, Watermark>(
                 trigger.strategy.watermarkCompareTo,
                 nextPoll.latestWatermark,
                 nextPoll.greatestWatermark,
                 paginatedResponseToPageTriggerItemsTransformed(ok.value, trigger.strategy)
             ))
         case failure is ResultErrorType -> failure
    }
}

/**
* Transforms a trigger by applying a function on its failed results
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `trigger` | Trigger<TriggerInputType, OperationInputType, OperationOutputType, OperationNextPageInputType, ResultErrorType, ConnectionType, TriggerResultItemType, OperationResultItemType, Watermark> | The trigger to transform
* | `mapper` | &#40;ResultErrorType&#41; &#45;&#62; NewResultErrorType | A function that receives the error and transforms it to a _better_ error
* |===
*/
fun mapOutputFailureTrigger<TriggerInputType, OperationInputType, OperationOutputType, OperationNextPageInputType, ResultErrorType <: ResultFailure, ConnectionType, TriggerResultItemType, OperationResultItemType, Watermark, NewResultErrorType>(
  trigger: Trigger<TriggerInputType, OperationInputType, OperationOutputType, OperationNextPageInputType, ResultErrorType, ConnectionType, TriggerResultItemType, OperationResultItemType, Watermark>,
  mapper: (ResultErrorType) -> NewResultErrorType)
  : Trigger<TriggerInputType, OperationInputType, OperationOutputType, OperationNextPageInputType, NewResultErrorType, ConnectionType, TriggerResultItemType, OperationResultItemType, Watermark> = do {
        trigger update {
          case op at .operation -> mapOutputFailureOperation(op, mapper)
        }
}
