/*
 * Copyright (c) 2025, Salesforce, Inc.,
 * All rights reserved.
 * For full license text, see the LICENSE.txt file
 */
%dw 2.7

import * from com::mulesoft::connectivity::Model
import location from dw::Runtime
import * from dw::core::Types
import * from dw::core::Arrays

/******************************************************************************************************************
* Metadata type builder
*******************************************************************************************************************/
fun createObjectTypeValueDefinition(name: String) : ObjectTypeValueDefinition = {
    name: name,
    "type": "Object",
    fields: []
}

fun createExtendedObjectTypeValueDefinition<T <: Object>(name: String, t : Type<T>) : ExtendedObjectTypeValueDefinition = {
  name: name,
  "type": "ExtendedObject",
  typeReference : location(t).sourceIdentifier as String ++ "::" ++ (t as String),
  customFields : []
}

fun createArrayTypeValueDefinition(definition : TypeValueDefinition) : ArrayTypeValueDefinition = {
  "type": "Array",
  item : definition
}

fun withName(a : ArrayTypeValueDefinition, name : String) : ArrayTypeValueDefinition = {
    "type": "Array",
    item : a.item,
    name: name
}

fun createStringTypeValueDefinition() : StringTypeValueDefinition = {
  "type" : "String"
}

fun createNumberTypeValueDefinition() : NumberTypeValueDefinition = {
  "type" : "Number"
}

fun createBooleanTypeValueDefinition() : BooleanTypeValueDefinition = {
  "type" : "Boolean"
}

fun createNullTypeValueDefinition() : NullTypeValueDefinition = {
  "type" : "Null"
}

fun enumOfValues(string: StringTypeValueDefinition, v: Array<String | EnumValue<String>>): StringTypeValueDefinition = do {
    var values = v map ($ match {
        case is String -> { value: $, label: $ }
        case is EnumValue<String> -> $
    })
    ---
    string update {
      case .values! -> ($ default []) ++ values
    }
}

fun withFormat(string: StringTypeValueDefinition, format: StringFormat) : StringTypeValueDefinition =
    string update {
    	case .format! -> format
    }

fun enumOfValues(number: NumberTypeValueDefinition, v: Array<Number | EnumValue<Number>>) : NumberTypeValueDefinition = do {
    var values = v map ($ match {
        case is Number -> { value: $, label: $ as String }
        case is EnumValue<Number> -> $
    })
    ---
    number update {
      case .values! -> ($ default []) ++ values
    }
}

fun withFormat(number: NumberTypeValueDefinition, format: NumberFormat) : NumberTypeValueDefinition =
    number update {
        case .format! -> format
    }

fun createUnionTypeValueDefinition() : UnionTypeValueDefinition = {
    "type": "Union",
    types: []
}

fun of(union: UnionTypeValueDefinition, value: TypeValueDefinition) : UnionTypeValueDefinition = {
    "type": union."type",
    types: union.types << value
}

fun of(union: UnionTypeValueDefinition, values: Array<TypeValueDefinition>) : UnionTypeValueDefinition = {
    "type": union."type",
    types: values
}

fun createField(name : String, definition : TypeValueDefinition, required : Boolean = true, repeated : Boolean = false) : ObjectFieldValueDefinition = {
    name : {
        localName : name,
        namespace: null
    },
    attributes: [],
    required: required,
    repeated: repeated,
    value : definition
}

fun withLabel(field: ObjectFieldValueDefinition, label : String) : ObjectFieldValueDefinition =
    withAnnotation(field, "label", label)

fun withDescription(field: ObjectFieldValueDefinition, description : String) : ObjectFieldValueDefinition =
    withAnnotation(field, "description", description)

fun withAnnotation(field: ObjectFieldValueDefinition, name: String, value: Any) : ObjectFieldValueDefinition =
    field update {
    	case annotations at .annotations! -> annotations default {} update { case .`$(name)`! -> value }
    }

fun withField(instance : ObjectTypeValueDefinition, field: ObjectFieldValueDefinition) : ObjectTypeValueDefinition = {
  name : instance.name,
  "type": "Object",
  fields : instance.fields << field
}

fun withFields(instance : ObjectTypeValueDefinition, fields: Array<ObjectFieldValueDefinition>) : ObjectTypeValueDefinition = {
  name : instance.name,
  "type": "Object",
  fields : instance.fields ++ fields
}

fun withField(instance : ExtendedObjectTypeValueDefinition, field: ObjectFieldValueDefinition) : ExtendedObjectTypeValueDefinition = {
  name: instance.name,
  typeReference : instance.typeReference,
  "type": "ExtendedObject",
  customFields : instance.customFields << field
}

fun withFields(instance : ExtendedObjectTypeValueDefinition, fields: Array<ObjectFieldValueDefinition>) : ExtendedObjectTypeValueDefinition = {
  name: instance.name,
  typeReference : instance.typeReference,
  "type": "ExtendedObject",
  customFields : instance.customFields ++ fields
}



var types_primitive = [String, Number, Boolean, Null, Date, DateTime]

fun createInt32() : NumberTypeValueDefinition = withFormat(createNumberTypeValueDefinition(), "int32")
fun createInt64() : NumberTypeValueDefinition = withFormat(createNumberTypeValueDefinition(), "int64")
fun createInteger() : NumberTypeValueDefinition = withFormat(createNumberTypeValueDefinition(), "integer")
fun createFloat() : NumberTypeValueDefinition = withFormat(createNumberTypeValueDefinition(), "float")
fun createDouble() : NumberTypeValueDefinition = withFormat(createNumberTypeValueDefinition(), "double")
fun createLWDate() : StringTypeValueDefinition = withFormat(createStringTypeValueDefinition(), "date")
fun createLWDateTime() : StringTypeValueDefinition = withFormat(createStringTypeValueDefinition(), "date-time")

var primitiveCreation = {
    String: createStringTypeValueDefinition,
    Boolean : createBooleanTypeValueDefinition,
    Null : createNullTypeValueDefinition,
    Number : createNumberTypeValueDefinition,
    Int32: createInt32,
    Int64: createInt64,
    Integer : createInteger,
    Float : createFloat,
    Double: createDouble,
    LWDate: createLWDate,
    Date: createLWDate,
    LWDateTime: createLWDateTime,
    DateTime: createLWDateTime
}

fun transform(t: Type) : TypeValueDefinition =
    if (types_primitive contains(t))
        primitiveCreation[t]()
    else if (isArrayType(t))
        createArray(t)
    else if (isUnionType(t))
        createUnionType(t)
    else
        createObject(t)

fun createArray(t : Type) : ArrayTypeValueDefinition = do {
    var array_T = transform(arrayItem(t))
    var notAlias = (t as String) == "Array"
    var def = createArrayTypeValueDefinition(array_T)
    ---
    if (notAlias) def else withName(def, t as String)
}

fun createObject(t: Type) : ObjectTypeValueDefinition = do {
    var fields = objectFields(t)
                    map ((item, index) ->
                    createField(
                        item.key.name.localName,
                        transform(item.value),
                        item.required,
                        item.repeated))
    ---
    createObjectTypeValueDefinition(objectName(t as String)) withFields fields
}

fun objectName(name : String) = if (name == "Object") "" else name

fun createUnionType(t : Type) : TypeValueDefinition = do {
    var union = createUnionTypeValueDefinition()
    var areLiterals = unionItems(t) every ((item) -> isLiteralType(item))
    var elements = unionItems(t) map ((item, index) ->
            transform(item)
    )
    ---
    if (areLiterals)
        createEnum(t)
    else
        of(union, elements)
}

fun createEnum(t: Type) : TypeValueDefinition = do {
    var literalType = typeOf(literalValueOf(unionItems(t)[0]))
    var isString = isStringType(literalType)
    var isNumber = isNumberType(literalType)
    var values = unionItems(t) map ((item, index) -> literalValueOf(item))
    ---
    if (isString)
        enumOfValues(transform(literalType) as StringTypeValueDefinition, values as Array<String>)
    else
        enumOfValues(transform(literalType) as NumberTypeValueDefinition, values as Array<Number>)
}