/*
 * Copyright (c) 2025, Salesforce, Inc.,
 * All rights reserved.
 * For full license text, see the LICENSE.txt file
 */
%dw 2.7
import * from com::mulesoft::connectivity::Model
import * from com::mulesoft::connectivity::decorator::Executor
import * from com::mulesoft::connectivity::decorator::Operation
import * from com::mulesoft::connectivity::transport::Http

/**
* Transform an operation into a offset paginated one specifying how to handle the offset parameter.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `operation` | Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType> | The operation to paginate
* | `itemsResolver` | (OpResultType) -> Array<ItemType> | A function that transforms the output into an array of items
* | `paramToOffsetFn` | (OpParam) -> Number | A function that retrieves the offset parameter from the input operation
* | `offsetToParamFn` | (OpParam, Number) -> OpParam | A function that updates the input operation with the new offset
* |===
*
*/
fun offsetPaginated<OpParam, OpResultType, OpResultErrorType<: ResultFailure, OpConnectionType, ItemType>(
    operation: Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType>,
    itemsResolver: (OpResultType) -> Array<ItemType>,
    paramToOffsetFn: (OpParam) -> Number,
    offsetToParamFn: (OpParam, Number) -> OpParam
): PaginatedOperation<OpParam, OpParam, Page<ItemType, OpParam>, OpResultErrorType, OpConnectionType> =
      (operation paginated (param, page) -> do {
         var items = itemsResolver(page)
         ---
         {
           items: items,
           (nextPage: {
                  args: offsetToParamFn(param, paramToOffsetFn(param) + sizeOf(items))
                 }) if !isEmpty(items)
         }
      })

/**
* Transform an operation into a page number paginated one specifying how to handle the page number parameter.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `operation` | Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType> | The operation to paginate
* | `itemsResolver` | (OpResultType) -> Array<ItemType> | A function that transforms the output into an array of items
* | `paramToPageNumberFn` | (OpParam) -> Number | A function that retrieves the page number parameter from the input operation
* | `pageNumberToParamFn` | (OpParam, Number) -> OpParam | A function that updates the input operation with the new page number
* |===
*
*/
fun pageNumberPaginated<OpParam, OpResultType, OpResultErrorType<: ResultFailure, OpConnectionType, ItemType>(
    operation: Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType>,
    itemsResolver: (OpResultType) -> Array<ItemType>,
    paramToPageNumberFn: (OpParam) -> Number,
    pageNumberToParamFn: (OpParam, Number) -> OpParam
): PaginatedOperation<OpParam, OpParam, Page<ItemType, OpParam>, OpResultErrorType, OpConnectionType> =
      (operation paginated (param, page) -> do {
           var items = itemsResolver(page)
           ---
           {
             items: items,
             (nextPage: {
                    args: pageNumberToParamFn(param, paramToPageNumberFn(param) + 1)
                   }) if !isEmpty(items)
           }
      })

@Internal(permits=[])
fun cursorBasedPaginated<OpParam, OpResultType, OpResultErrorType<: ResultFailure, OpConnectionType, ItemType>(
    operation: Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType>,
    itemsResolver: (OpResultType) -> Array<ItemType>,
    fromPageToCursorStringFn: (OpResultType) -> String,
    fromCursorAndParamToParamFn: (OpParam, String) -> OpParam
): PaginatedOperation<OpParam, OpParam, Page<ItemType, OpParam>, OpResultErrorType, OpConnectionType> =
      (operation paginated (param, page) -> do {
           var items = itemsResolver(page)
           var cursor = fromPageToCursorStringFn(page)
           ---
           {
             items: items,
             (nextPage: {
                    args: fromCursorAndParamToParamFn(param, cursor)
                   }) if (!isEmpty(items) and !isEmpty(cursor))
           }
      })

/**
* Transform an operation into a marker paginated one specifying how to handle the marker parameter.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `operation` | Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType> | The operation to paginate
* | `itemsResolver` | (OpResultType) -> Array<ItemType> | A function that transforms the output into an array of items
* | `pageToMarkerFn` | (OpResultType) -> String | A function that retrieves the marker parameter from the response body
* | `markerToParamFn` | (OpParam, String) -> OpParam | A function that updates the input operation with the new marker
* |===
*
*/
fun markerPaginated<OpParam, OpResultType, OpResultErrorType<: ResultFailure, OpConnectionType, ItemType>(
    operation: Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType>,
    itemsResolver: (OpResultType) -> Array<ItemType>,
    pageToMarkerFn: (OpResultType) -> String,
    markerToParamFn: (OpParam, String) -> OpParam
): PaginatedOperation<OpParam, OpParam, Page<ItemType, OpParam>, OpResultErrorType, OpConnectionType> =
      cursorBasedPaginated(operation,itemsResolver,pageToMarkerFn,markerToParamFn)

/**
* Transform an operation into a hypermedia paginated one specifying how to get the next URL.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `operation` | Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType> | The operation to paginate
* | `itemsResolver` | (OpResultType) -> Array<ItemType> | A function that transforms the output into an array of items
* | `pageToHypermediaFn` | (OpResultType) -> String | A function that retrieves the next URL parameter from the response body
* | `hypermediaToParamFn` | (OpParam, String) -> OpParam | A function that updates the input operation with the new hypermedia parameters updated
* |===
*
*/
fun hypermediaPaginated<OpParam, OpResultType, OpResultErrorType<: ResultFailure, OpConnectionType, ItemType>(
    operation: Operation<OpParam, OpResultType, OpResultErrorType, OpConnectionType>,
    itemsResolver: (OpResultType) -> Array<ItemType>,
    pageToHypermediaFn: (OpResultType) -> String,
    hypermediaToParamFn: (OpParam, String) -> OpParam
): PaginatedOperation<OpParam, OpParam, Page<ItemType, OpParam>, OpResultErrorType, OpConnectionType> =
      cursorBasedPaginated(operation,itemsResolver,pageToHypermediaFn,hypermediaToParamFn)
